# -*- coding: utf-8 -*-

from passport.backend.vault.api.errors import ExactlyOneFieldError
from passport.backend.vault.api.views import validators
from passport.backend.vault.api.views.base_form import (
    BaseForm,
    FieldList,
    IntegerField,
    PaginationFormMixin,
    SecretUUIDField,
    StringField_,
    VersionUUIDField,
)
from wtforms import (
    BooleanField,
    FormField,
    StringField,
)


class ListTokensForm(BaseForm, PaginationFormMixin):
    with_revoked = BooleanField(
        'with_revoked',
        description='add revoked tokens to the list',
        default=False,
        false_values=('false', '0', '', 0, False),
    )


class CreateTokenForm(BaseForm):
    tvm_client_id = IntegerField(
        'Integer',
        description='tvm vault_client id',
        default=None,
        validators=[
            validators.Optional(),
        ],
    )
    signature = StringField(
        'Token signature',
        description='token signature',
        validators=[
            validators.Optional(),
            validators.Length(max=255),
        ],
    )
    comment = StringField_(
        'Comment',
        description='comment',
        strip=True,
        validators=[
            validators.Optional(),
            validators.Length(max=1023),
        ],
    )


class GetTokenInfoForm(BaseForm):
    token_uuid = StringField(
        'Token uuid',
        description='Token',
        validators=[
            validators.Optional(),
        ],
    )

    token = StringField(
        'Token',
        description='token',
        validators=[
            validators.Optional(),
        ],
    )

    def validate(self):
        if not super(GetTokenInfoForm, self).validate():
            return False
        fields = [self.token_uuid, self.token]
        set_fields = len(filter(lambda x: not x.is_missing, fields))
        if set_fields != 1:
            raise ExactlyOneFieldError(['token_uuid', 'token'])
        return True


class TokenizedRevokeTokenForm(BaseForm):
    token = StringField(
        'Delegation token',
        description='Delegation token',
        validators=[validators.DataRequired()],
    )

    service_ticket = StringField(
        'Service ticket',
        description='TVM 2.0 Service ticket',
        validators=[validators.DataRequired()],
    )

    signature = StringField(
        'Delegation token signature',
        description='Delegation token signature',
        validators=[
            validators.Optional(),
            validators.Length(max=128),
        ],
    )

    secret_uuid = SecretUUIDField(
        'Secret UUID',
        description='if not empty then validate whether token is created for passed secret uuid',
        validators=[
            validators.Optional(),
        ],
    )


class TokenizedRevokeTokensForm(BaseForm):
    tokenized_requests = FieldList(
        FormField(
            TokenizedRevokeTokenForm
        ),
        description='Tokenized requests (a list of no more than 100 items)',
    )


class TokenizedRequestForm(BaseForm):
    secret_version = VersionUUIDField(
        'Secret version UUID',
        description='secret version',
        validators=[
            validators.Optional(),
        ],
    )

    token = StringField(
        'Delegation token',
        description='Delegation token',
        validators=[
            validators.DataRequired(),
        ],
    )

    service_ticket = StringField(
        'Service ticket',
        description='TVM 2.0 Service ticket',
        validators=[
            validators.Optional(),
        ],
    )

    signature = StringField(
        'Token signature',
        description='Delegation token signature',
        validators=[
            validators.Optional(),
            validators.Length(max=255),
        ],
    )

    secret_uuid = SecretUUIDField(
        'Secret UUID',
        description='if not empty then validate whether token is created for passed secret uuid',
        validators=[
            validators.Optional(),
        ],
    )

    uid = IntegerField(
        'User UID',
        description='if not empty check that the user has the right to read the secret',
        validators=[
            validators.Optional(),
        ],
    )


class TokenizedRequestsForm(BaseForm):
    tokenized_requests = FieldList(
        FormField(
            TokenizedRequestForm
        ),
        description='Tokenized requests (a list of no more than 100 items)',
    )
