# -*- coding: utf-8 -*-

import json

from passport.backend.utils import common
from passport.backend.vault.api import tvm
from passport.backend.vault.api.db import get_db
from passport.backend.vault.api.errors import (
    AccessError,
    BaseError,
    DelegationTokenDoesNotMatchSecretError,
    DelegationTokenMismatchTvmAppError,
    DelegationTokenRevokedError,
    InvalidDelegationTokenSignatureError,
    NonexistentEntityError,
    ServiceTicketParsingError,
    ServiceTicketRequiredError,
    TokenNotAssociatedWithTvmAppError,
    ValidationError,
)
from passport.backend.vault.api.models import (
    DelegationToken,
    ExternalRecord,
    Roles,
    Secret,
    SecretVersion,
    UserRole,
)
from passport.backend.vault.api.models.base import (
    State,
    TokenState,
)
from passport.backend.vault.api.models.secret import SecretUUIDType
from passport.backend.vault.api.models.secret_version import SecretVersionUUIDType
from passport.backend.vault.api.utils import ulid
from passport.backend.vault.api.value_manager import (
    get_value_manager,
    ValueType,
)
from passport.backend.vault.api.views.secrets.views import BaseSecretView
from passport.backend.vault.api.views.tokens.forms import (
    CreateTokenForm,
    GetTokenInfoForm,
    ListTokensForm,
    TokenizedRequestsForm,
    TokenizedRevokeTokensForm,
)
from passport.backend.vault.api.views.versions.views import BaseVersionView
from sqlalchemy import (
    and_,
    func,
    join,
    literal_column,
    or_,
    select,
    union,
)
from ticket_parser2 import ServiceTicket
from ticket_parser2.exceptions import TicketParsingException


class ListTokensView(BaseSecretView):
    """
    Получить список делегирующих токенов
    -----
    returns = ['tokens']
    raises = []
    example = {
        'arguments': {'<string:secret_uuid>': 'sec-0000000000000000000000ygj0'},
        'response': {
            'tokens': [{
                'created_at': 1445385600.0,
                'created_by': 100,
                'creator_login': 'vault-test-100',
                'signature': '123',
                'secret_uuid': 'sec-0000000000000000000000ygj0',
                'state_name': 'normal',
                'token_uuid': 'tid-0000000000000000000000nbk0',
                'tvm_app': {
                    'abc_department': {
                        'display_name': u'\u041f\u0430\u0441\u043f\u043e\u0440\u0442',
                        'id': 14,
                        'unique_name': 'passp'
                    },
                    'abc_state': 'granted',
                    'abc_state_display_name': u'\u0412\u044b\u0434\u0430\u043d',
                    'name': 'social api (dev)',
                    'tvm_client_id': 2000367
                },
                'tvm_client_id': 2000367
            }],
            'status': 'ok',
        },
    }
    """
    form = ListTokensForm
    statbox_mode = 'list_tokens'
    use_slave = True

    def process_request(self, secret_uuid):
        page = self.processed_form.page.data
        page_size = self.processed_form.page_size.data

        # Проверить существование секрета
        secret = Secret.get_by_id(secret_uuid)
        self.check_if_has_role(role=[Roles.READER, Roles.APPENDER], secret_uuid=secret_uuid)

        filters = (DelegationToken.state == TokenState.normal.value)
        if self.processed_form.with_revoked.data:
            filters = filters | (DelegationToken.state == TokenState.revoked.value)

        self.response_values.update({
            'tokens': map(
                lambda x: x.serialize(max_depth=3),
                DelegationToken.query.filter(
                    DelegationToken.secret_uuid == secret.uuid,
                    filters,
                ).order_by(
                    DelegationToken.id
                ).limit(
                    page_size
                ).offset(
                    page * page_size
                ).all(),
            ),
        })


class CreateTokenView(BaseSecretView):
    """
    Получить новый делегирующий токен
    -----
    returns = ['token', 'token_uuid']
    raises = [NonexistentEntityError]
    example = {
        'arguments': {'<string:secret_uuid>': 'sec-0000000000000000000000ygj0'},
        'data': 'signature=153&tvm_client_id=2000367',
        'response': {
            'token': 'gcwCk8kPz65Y54HnTrDKbjAN0FXAJPlnJVzpJtpRhwE.1.c318dcccb19a8765',
            'token_uuid': 'tid-0000000000000000000000ygj2',
            'tvm_app': {
                u'abc_department': {
                    u'display_name': u'\u041f\u0430\u0441\u043f\u043e\u0440\u0442',
                    u'id': 14,
                    u'unique_name': u'passp',
                },
                u'abc_state': u'granted',
                u'abc_state_display_name': u'\u0412\u044b\u0434\u0430\u043d',
                u'name': u'social api (dev)',
                u'tvm_client_id': 2000367,
            },
            'secret_uuid': 'sec-0000000000000000000000ygj0',
            'status': 'ok',
        },
    }
    """
    form = CreateTokenForm
    statbox_mode = 'create_token'

    def process_request(self, secret_uuid):
        # Проверить существование секрета
        Secret.get_by_id(secret_uuid)
        self.check_if_has_role(role=Roles.READER, secret_uuid=secret_uuid)

        token, delegation_token = DelegationToken.create_token(
            created_by=self.validated_uid,
            secret_uuid=secret_uuid,
            tvm_client_id=self.processed_form.tvm_client_id.data,
            signature=self.processed_form.signature.data,
            comment=self.processed_form.comment.data,
        )
        self.commit(delegation_token)

        self.statbox_log(
            action='success',
            secret_uuid=str(delegation_token.secret_uuid),
            token_uuid=str(delegation_token.id),
            tvm_id=delegation_token.tvm_client_id,
            comment=delegation_token.comment,
            uid=delegation_token.created_by,
        )

        self.response_values.update(common.noneless_dict({
            'token': token,
            'token_uuid': delegation_token.id,
            'tvm_app': delegation_token.tvm_app.serialize(max_depth=3) if delegation_token.tvm_app is not None else None,
            'secret_uuid': delegation_token.secret_uuid,
        }))


class GetTokenInfoView(BaseSecretView):
    """
    Получить информацию о токене по токену или token_uuid. В ручку можно передать либо токен либо token_uid.
    -----
    returns = ['token_info', 'owners', 'readers']
    raises = [InvalidDelegationTokenError, NonexistentEntityError, ExactlyOneFieldError]
    example = {
        'data': 'token_uuid=tid-0000000000000000000008jn22',
        'response': {
            u'owners': [{
                u'created_at': 1445385600.0,
                u'created_by': 100,
                u'creator_login': u'vault-test-100',
                u'login': u'vault-test-100',
                u'role_slug': u'OWNER',
                u'uid': 100
            }],
            u'readers': [{
                u'created_at': 1445385600.0,
                u'created_by': 100,
                u'creator_login': u'vault-test-100',
                u'login': u'vault-test-100',
                u'role_slug': u'OWNER',
                u'uid': 100
            }],
            u'status': u'ok',
            u'token_info': {
                u'created_at': 1445385600.0,
                u'created_by': 100,
                u'creator_login': u'vault-test-100',
                u'secret_uuid': u'sec-0000000000000000000008jn20',
                u'state_name': u'normal',
                u'token_uuid': u'tid-0000000000000000000008jn22',
                u'tvm_app': {
                    u'abc_department': {
                        u'display_name': u'\u041f\u0430\u0441\u043f\u043e\u0440\u0442',
                        u'id': 14,
                        u'unique_name': u'passp',
                    },
                    u'abc_state': u'granted',
                    u'abc_state_display_name': u'\u0412\u044b\u0434\u0430\u043d',
                    u'name': u'social api (dev)',
                    u'tvm_client_id': 2000367,
                },
                u'tvm_client_id': 2000367,
            },
        },
    }
    """
    form = GetTokenInfoForm
    statbox_mode = 'token_info'
    required_user_auth = False
    use_slave = True

    def _get_filters(self):
        token_uuid = self.processed_form.token_uuid.data
        if token_uuid is not None:
            return (DelegationToken.id == token_uuid)

        token = self.processed_form.token.data
        if token is not None:
            DelegationToken.validate_token(token)
            return (DelegationToken.hashed_token == DelegationToken.hash_token(token))

    def process_request(self):
        delegation_token = DelegationToken.query.filter(
            self._get_filters(),
        ).one_or_none()
        if delegation_token is None:
            raise NonexistentEntityError()

        secret_roles = delegation_token.secret.one().secret_roles
        owners = filter(
            lambda x: x.role_id == Roles.OWNER,
            secret_roles,
        )
        readers = filter(
            lambda x: x.role_id in (Roles.READER, Roles.OWNER,),
            secret_roles,
        )

        self.statbox_log(
            action='success',
            token_uuid=str(delegation_token.id),
        )

        self.response_values.update(common.noneless_dict({
            'token_info': delegation_token.serialize(
                max_depth=3,
                exclude=['signature'],
            ),
            'owners': owners,
            'readers': readers,
        }))


class RevokeTokenView(BaseSecretView):
    """
    Отозвать токен по token_uuid
    -----
    returns = []
    raises = [NonexistentEntityError]
    example = {
        'arguments': {'<string:token_uuid>': 'tid-0000000000000000000000ygj2'},
        'response': {
            'status': 'ok',
        },
    }
    """
    statbox_mode = 'revoke_token'

    def process_request(self, token_uuid):
        delegation_token = DelegationToken.get_by_id(token_uuid)
        self.check_if_has_role(role=Roles.OWNER, secret_uuid=delegation_token.secret_uuid)

        if delegation_token.state == TokenState.normal.value:
            delegation_token.revoke(revoked_by=self.validated_uid)
            self.commit(delegation_token)

            self.statbox_log(
                action='success',
                source='owner',
                token_uuid=str(delegation_token.id),
                uid=delegation_token.revoked_by,
            )


class RestoreTokenView(BaseSecretView):
    """
    Восстановить отозванный токен по token_uuid
    -----
    returns = []
    raises = [NonexistentEntityError]
    example = {
        'arguments': {'<string:token_uuid>': 'tid-0000000000000000000000ygj2'},
        'response': {
            'status': 'ok',
        },
    }
    """
    statbox_mode = 'restore_token'

    def process_request(self, token_uuid):
        delegation_token = DelegationToken.get_by_id(token_uuid)
        self.check_if_has_role(role=Roles.OWNER, secret_uuid=delegation_token.secret_uuid)

        if delegation_token.state == TokenState.revoked.value:
            delegation_token.restore()
            self.commit(delegation_token)

            self.statbox_log(
                action='success',
                source='owner',
                token_uuid=str(delegation_token.id),
                uid=self.validated_uid,
            )


class BaseTokenizedView(BaseSecretView, BaseVersionView):
    required_user_auth = False
    max_tokenized_request = 100

    def __init__(self):
        super(BaseTokenizedView, self).__init__()
        self.processed_form_errors = dict()

    def process_form(self, form):
        processed_form = super(BaseTokenizedView, self).process_form(form, skip_exception=True)
        self.processed_form_errors = dict(map(
            lambda x: (x['index'], x),
            processed_form.errors.get('tokenized_requests', []),
        ))

        requests_count = len(processed_form.tokenized_requests)
        if requests_count < 1 or requests_count > self.max_tokenized_request:
            raise ValidationError(dict(
                tokenized_requests='size_1_{}'.format(self.max_tokenized_request),
            ))

        return processed_form

    def check_user_access(self):
        self.log_consumer_tvm_app()

    def log_consumer_tvm_app(self):
        """Пишем в лог параметры приложения, если нам передали заголовок X-Ya-Service-Ticket"""
        self.statbox_auth_info.update(dict(
            has_x_ya_service_ticket=self.service_ticket is not None,
        ))
        service_ticket = self.service_ticket
        if service_ticket is not None:
            try:
                parsed_service_ticket = tvm.get_service_context().check(
                    service_ticket.encode('utf-8')
                )
                source_tvm_client_id = parsed_service_ticket.src
                self.statbox_auth_info.update(dict(
                    auth_type='service_ticket',
                    auth_service_ticket=ServiceTicket.remove_signature(service_ticket),
                    auth_tvm_app_id=source_tvm_client_id,
                ))
            except TicketParsingException:
                pass

    def check_service_ticket(self, service_ticket, tvm_client_id):
        if not service_ticket:
            raise ServiceTicketRequiredError()
        try:
            return tvm.get_service_context().check(service_ticket.encode('utf-8')).src == tvm_client_id
        except TicketParsingException as ex:
            raise ServiceTicketParsingError(
                ticket_status=ex.status.value,
                ticket_message=ex.message,
            )

    def check_token_secret_uuid(self, requested_secret_uuid, secret_uuid):
        if requested_secret_uuid is None:
            return

        requested_secret_uuid = ulid.ULID(
            requested_secret_uuid,
            prefix=SecretUUIDType.prefix,
        )
        if requested_secret_uuid != secret_uuid:
            raise DelegationTokenDoesNotMatchSecretError()


class TokenizedRevokeTokensView(BaseTokenizedView):
    """
    Отозвать токен от имени tvm-приложения
    Принимает список запросов с токенами, сервис-тикетом и подписью,
    если они требуются для токена.
    Возвращает список с результатом отзыва для каждого запроса с токеном.
    Если при отзыве токена произошла ошибка, то вернет в элементе списка "status": "error".
    Код ошибки и сообщение вернет в полях "code" и "message".
    Для validation_error в ответе добавляем словарик errors с ошибками валидации подзапросов.
    Возможные коды ошибок для списка с ответами:
    "invalid_token",
    "nonexistent_entity_error",
    "service_ticket_parsing_error",
    "service_ticket_required_error",
    "token_not_associated_with_tvm_app",
    "validation_error".

    Потребитель должен передавать в ручку параметр consumer, содержащий A.B, где A — имя потребителя,
    которое согласовывается вместе с квотой на ручку, B — версия бинарника, который будет ходить в ручку.
    -----
    returns = ['result']
    raises = []
    example = {
        'headers': {'Content-Type': 'application/json'},
        'data': {
            'tokenized_requests': [
                {
                    'token': 'gcwCk8kPz65Y54HnTrDKbjAN0FXAJPlnJVzpJtpRhwE.1.c318dcccb19a8765',
                    'service_ticket': '3:serv:aBcDeF',
                    'signature': '123',
                },
            ],
        },
        'response': {
            'result': [
                {
                    'token': 'gcwCk8kPz65Y54HnTrDKbjAN0FXAJPlnJVzpJtpRhwE.1.c318dcccb19a8765',
                    'status': 'ok',
                },
            ],
            'status': 'ok',
        },
    }
    """
    form = TokenizedRevokeTokensForm
    statbox_mode = 'tokenized_revoke_tokens'

    def check_token_access(self, token, service_ticket, signature):
        if (
            token.tvm_client_id is not None
            and not self.check_service_ticket(service_ticket, token.tvm_client_id)
        ):
            raise DelegationTokenMismatchTvmAppError()

        if token.signature is not None and token.signature != signature:
            raise InvalidDelegationTokenSignatureError()

        if token.signature is None and signature is not None:
            self.statbox_extra_info['skipped_signature_check'] = signature

        return True

    def _get_tokens(self):
        tokens = [
            DelegationToken.hash_token(tokenized_request.token.data)
            for index, tokenized_request in enumerate(self.processed_form.tokenized_requests.entries)
            if index not in self.processed_form_errors
        ]
        if not tokens:
            return dict()

        result = DelegationToken.query.filter(
            DelegationToken.hashed_token.in_(tokens),
        ).all()
        return {t.hashed_token: t for t in result}

    def _revoke_tokens(self):
        values = []
        found_tokens = self._get_tokens()

        db_session = get_db().session

        for index, tokenized_request in enumerate(self.processed_form.tokenized_requests.entries):
            try:
                hashed_token = DelegationToken.hash_token(tokenized_request.token.data) if tokenized_request.token.data else ''

                if index in self.processed_form_errors:
                    raise ValidationError(
                        self.processed_form_errors[index],
                    )

                token = found_tokens.get(hashed_token)

                if token:
                    try:
                        if token.tvm_client_id is None:
                            raise TokenNotAssociatedWithTvmAppError()

                        self.check_token_secret_uuid(
                            tokenized_request.secret_uuid.data,
                            token.secret_uuid,
                        )

                        self.check_token_access(
                            token,
                            service_ticket=tokenized_request.service_ticket.data,
                            signature=tokenized_request.signature.data,
                        )
                    except BaseError as e:
                        e.kwargs['token_uuid'] = str(token.id)
                        raise e

                    if token.state == TokenState.normal.value:
                        token.revoke()
                        db_session.add(token)

                        self.statbox_log(
                            action='success',
                            source='tvm',
                            token_uuid=str(token.id),
                            tvm_client_id=token.tvm_client_id,
                        )

                    values.append({
                        'token': tokenized_request.token.data,
                        'token_uuid': token.id,
                        'status': 'ok',
                    })
                else:
                    DelegationToken.validate_token(tokenized_request.token.data)
                    raise NonexistentEntityError()
            except BaseError as error:
                values.append(common.noneless_dict(common.merge_dicts(
                    error.as_dict(),
                    dict(
                        token=tokenized_request.token.data,
                        secret_uuid=tokenized_request.secret_uuid.data,
                    ),
                )))
                self.statbox_log(**common.merge_dicts(
                    error.as_dict(),
                    dict(
                        action='error',
                        source='tvm',
                        hashed_token=hashed_token,
                    ),
                ))

        db_session.commit()
        return values

    def process_request(self):
        self.response_values.update({
            'result': self._revoke_tokens(),
        })


class TokenizedRequestsView(BaseTokenizedView):
    """
    Получить версию секрета по делегирующему токену.
    Принимает список запросов с токенами, сервис-тикетом и подписью,
    если они требуются для токена.
    Возвращает список с версиями секретов для каждого запроса с токеном.
    Если версию не удалось получить по токену, то вернет в элементе списка "status": "error".
    Код ошибки и сообщение вернет в полях "code" и "message".
    Для validation_error в ответе добавляем словарик errors с ошибками валидации подзапросов.
    Возможные коды ошибок для списка с ответами:
    "access_error",
    "delegation_access_error",
    "invalid_token",
    "nonexistent_entity_error",
    "service_ticket_parsing_error",
    "service_ticket_required_error",
    "token_does_not_match_secret",
    "token_not_associated_with_tvm_app",
    "token_revoked",
    "validation_error".

    Потребитель должен передавать в ручку параметр consumer, содержащий A.B, где A — имя потребителя,
    которое согласовывается вместе с квотой на ручку, B — версия бинарника, который будет ходить в ручку.
    -----
    returns = ['secrets']
    raises = []
    example = {
        'headers': {'Content-Type': 'application/json'},
        'data': {
            'tokenized_requests': [
                {
                    'token': 'gcwCk8kPz65Y54HnTrDKbjAN0FXAJPlnJVzpJtpRhwE',
                    'signature': '123',
                },
                {
                    'token': 'gcwCk8kPz65Y54HnTrDKbjAN0FXAJPlnJVzpJtpRhwE',
                    'secret_version': 'ver-0000000000000000000000ygj2',
                    'service_ticket': '3:serv:aBcDeF',
                    'signature': '123',
                },
                {
                    'token': 'unknown-token',
                    'secret_version': 'ver-0000000000000000000000ygj2',
                    'service_ticket': '3:serv:aBcDeF',
                    'signature': '123',
                },
                {
                    'token': 'gcwCk8kPz65Y54HnTrDKbjAN0FXAJPlnJVzpJtpRhwE',
                    'secret_uuid': 'ver-12345',
                    'secret_version': 'ver-12345',
                    'service_ticket': '3:serv:aBcDeF',
                    'signature': '123',
                },
            ],
        },
        'response': {
            'secrets': [
                {
                    'token': 'gcwCk8kPz65Y54HnTrDKbjAN0FXAJPlnJVzpJtpRhwE',
                    'token_uuid': 'tid-0000000000000000000000ygj4',
                    'secret_version': 'ver-0000000000000000000000ygj3',
                    'status': 'ok',
                    'value': [{
                        'key': 'login',
                        'value': 'ppodolsky',
                    }, {
                        'key': 'password',
                        'value': '7890133',
                    }, {
                        'key': 'cert',
                        'value': 'U2VjcmV0IGZpbGU=',
                        'encoding': 'base64',
                    }],
                },
                {
                    'token': 'gcwCk8kPz65Y54HnTrDKbjAN0FXAJPlnJVzpJtpRhwE',
                    'token_uuid': 'tid-0000000000000000000000ygj4',
                    'secret_version': 'ver-0000000000000000000000ygj2',
                    'status': 'ok',
                    'value': [{
                        'key': 'login',
                        'value': 'ppodolsky',
                    }, {
                        'key': 'password',
                        'value': '123456',
                    }],
                },
                {
                    'code': 'invalid_token',
                    'message': 'Invalid delegation token',
                    'secret_version': 'ver-0000000000000000000000ygj2',
                    'status': 'error',
                    'token': 'unknown-token',
                },
                {
                    'code': 'validation_error',
                    'errors': {
                        'index': 3,
                        'secret_uuid': ['invalid_uuid_prefix'],
                        'secret_version': ['invalid_uuid_value'],
                    },
                    'message': 'Validation error',
                    'secret_uuid': 'ver-12345',
                    'secret_version': 'ver-12345',
                    'status': 'error',
                    'token': 'gcwCk8kPz65Y54HnTrDKbjAN0FXAJPlnJVzpJtpRhwE',
                },
            ],
            'status': 'ok',
        },
    }
    """
    form = TokenizedRequestsForm
    statbox_mode = 'tokenized_requests'
    use_slave = True

    def process_request(self):
        version_rows = self.get_rows()
        self.response_values.update({
            'secrets': version_rows,
        })

    def check_token_access(self, row, service_ticket, signature):
        if (
            row['tvm_client_id'] is not None
            and not self.check_service_ticket(service_ticket, row['tvm_client_id'])
        ):
            raise DelegationTokenMismatchTvmAppError()

        if row['signature'] is not None and row['signature'] != signature:
            raise InvalidDelegationTokenSignatureError()

        if row['signature'] is None and signature is not None:
            self.statbox_extra_info['skipped_signature_check'] = signature

        return True

    def check_uid_access(self, row, uid, secrets_for_users):
        if uid is not None and row['secret_uuid'] not in secrets_for_users.get(uid, set()):
            raise AccessError()

    def get_rows(self):
        values = []
        found_versions = self.request_versions_from_database()
        secrets_for_users = self.get_secrets_for_uids(found_versions)

        for index, tokenized_request in enumerate(self.processed_form.tokenized_requests.entries):
            try:
                hashed_token = DelegationToken.hash_token(tokenized_request.token.data) if tokenized_request.token.data else ''

                if index in self.processed_form_errors:
                    raise ValidationError(
                        self.processed_form_errors[index],
                    )

                requested_secret_version = tokenized_request.secret_version.data or ''
                if requested_secret_version:
                    requested_secret_version = ulid.ULID(
                        requested_secret_version,
                        prefix=SecretVersionUUIDType.prefix,
                    )

                tokenized_version = found_versions.get((hashed_token, requested_secret_version))
                if tokenized_version:
                    try:
                        if tokenized_version['token_state'] == TokenState.revoked.value:
                            raise DelegationTokenRevokedError()

                        self.check_token_secret_uuid(
                            tokenized_request.secret_uuid.data,
                            tokenized_version['secret_uuid'],
                        )

                        self.check_token_access(
                            tokenized_version,
                            service_ticket=tokenized_request.service_ticket.data,
                            signature=tokenized_request.signature.data,
                        )

                        self.check_uid_access(
                            tokenized_version,
                            tokenized_request.uid.data,
                            secrets_for_users,
                        )

                        constructed_value = common.noneless_dict({
                            'token': tokenized_request.token.data,
                            'token_uuid': str(tokenized_version['token_uuid']),
                            'secret_version': tokenized_version['secret_version'],
                            'value': tokenized_version['value'],
                            'comment': tokenized_version['comment'],
                            'status': 'ok',
                            'uid': tokenized_request.uid.data,
                        })

                        self.check_state(tokenized_version, constructed_value)
                        self.check_version_expiration(tokenized_version, constructed_value)
                    except BaseError as e:
                        e.kwargs['token_uuid'] = str(tokenized_version['token_uuid'])
                        raise e

                    values.append(constructed_value)

                    log_entry = dict(
                        action='success',
                        source='tvm',
                        token_uuid=str(tokenized_version['token_uuid']),
                        secret_version=str(tokenized_version['secret_version']),
                        tvm_client_id=tokenized_version['tvm_client_id'],
                    )
                    if tokenized_request.uid.data is not None:
                        log_entry['uid'] = tokenized_request.uid.data
                    self.statbox_log(**log_entry)
                else:
                    DelegationToken.validate_token(tokenized_request.token.data)
                    raise NonexistentEntityError()
            except BaseError as error:
                values.append(common.noneless_dict(common.merge_dicts(
                    error.as_dict(),
                    dict(
                        token=tokenized_request.token.data,
                        secret_version=tokenized_request.secret_version.data,
                        secret_uuid=tokenized_request.secret_uuid.data,
                        uid=tokenized_request.uid.data
                    ),
                )))
                log_entry = common.merge_dicts(
                    error.as_dict(),
                    dict(
                        action='error',
                        source='tvm',
                        hashed_token=hashed_token,
                        secret_version=str(tokenized_request.secret_version.data),
                    ),
                )
                if tokenized_request.uid.data is not None:
                    log_entry['uid'] = tokenized_request.uid.data
                self.statbox_log(**log_entry)

        return values

    def _get_versions_expressions(self):
        entries = [
            r for index, r in enumerate(self.processed_form.tokenized_requests.entries)
            if index not in self.processed_form_errors and r.secret_version.data is not None
        ]

        if not entries:
            return

        secret_conditions = []
        for tokenized_request in entries:
            secret_conditions.append(and_(
                DelegationToken.hashed_token == DelegationToken.hash_token(tokenized_request.token.data),
                SecretVersion.version == tokenized_request.secret_version.data,
            ))

        joined = join(DelegationToken, SecretVersion, DelegationToken.secret_uuid == SecretVersion.secret_uuid)
        joined = joined.join(Secret, SecretVersion.secret_uuid == Secret.uuid)

        expr = select(
            [
                DelegationToken.id.label('token_uuid'),
                DelegationToken.hashed_token.label('hashed_token'),
                DelegationToken.secret_uuid.label('secret_uuid'),
                SecretVersion.version.label('version'),
                SecretVersion.value_type.label('value_type'),
                SecretVersion._value.label('value'),
                SecretVersion.cipher_key_id.label('cipher_key_id'),
                DelegationToken.tvm_client_id.label('tvm_client_id'),
                DelegationToken.signature.label('signature'),
                SecretVersion.state.label('version_state'),
                Secret.state.label('secret_state'),
                SecretVersion.expired_at.label('expired_at'),
                literal_column('\'versioned\'').label('versioned'),
                DelegationToken.comment.label('comment'),
                DelegationToken.state.label('token_state'),
            ],
            or_(*secret_conditions),
            joined,
            use_labels=True,
        )
        return expr

    def _get_head_versions_expressions(self):
        entries = [
            r for index, r in enumerate(self.processed_form.tokenized_requests.entries)
            if index not in self.processed_form_errors and r.secret_version.data is None
        ]
        if not entries:
            return

        tokens = [DelegationToken.hash_token(t.token.data) for t in entries]

        last_versions = select(
            [
                DelegationToken.id.label('delegation_token_id'),
                func.max(SecretVersion.version).label('last_version'),
            ],
        ).where(
            and_(
                DelegationToken.hashed_token.in_(tokens),
                DelegationToken.secret_uuid == SecretVersion.secret_uuid,
                SecretVersion.state == State.normal.value,
            ),
        ).group_by(
            DelegationToken.id,
        ).alias('last_versions')

        expr = select(
            [
                DelegationToken.id.label('token_uuid'),
                DelegationToken.hashed_token.label('hashed_token'),
                DelegationToken.secret_uuid.label('secret_uuid'),
                SecretVersion.version.label('version'),
                SecretVersion.value_type.label('value_type'),
                SecretVersion._value.label('value'),
                SecretVersion.cipher_key_id.label('cipher_key_id'),
                DelegationToken.tvm_client_id.label('tvm_client_id'),
                DelegationToken.signature.label('signature'),
                SecretVersion.state.label('version_state'),
                Secret.state.label('secret_state'),
                SecretVersion.expired_at.label('expired_at'),
                literal_column('\'\'').label('versioned').label('versioned'),
                DelegationToken.comment.label('comment'),
                DelegationToken.state.label('token_state'),
            ],
            use_labels=True,
        ).where(
            and_(
                SecretVersion.version == last_versions.c.last_version,
                DelegationToken.id == last_versions.c.delegation_token_id,
                Secret.uuid == DelegationToken.secret_uuid,
            ),
        )
        return expr

    def request_versions_from_database(self):
        queries = filter(
            lambda x: x is not None,
            [
                self._get_versions_expressions(),
                self._get_head_versions_expressions(),
            ],
        )
        if not queries:
            return dict()

        rows = list(map(
            lambda x: (
                (x['hashed_token'], x['version'] if x['versioned'] == 'versioned' else ''),
                {
                    'token_uuid': x['token_uuid'],
                    'secret_uuid': x['secret_uuid'],
                    'secret_version': x['version'],
                    'value': json.loads(
                        get_value_manager(self.config).decode(
                            x['cipher_key_id'],
                            ValueType(x['value_type']),
                            x['value'],
                        ),
                    ),
                    'tvm_client_id': x['tvm_client_id'],
                    'signature': x['signature'],
                    'transitive_state': max(x['version_state'], x['secret_state']),
                    'expired_at': x['expired_at'],
                    'comment': x['comment'],
                    'token_state': x['token_state'],
                },
            ),
            get_db().session.execute(union(*queries)),
        ))
        return dict(rows)

    def get_secrets_for_uids(self, found_versions):
        """
        Если для токена указан uid, достаем из базы секреты с правом четния
        """
        # Отфильтровываем токены для которых надо искать роли
        entries = [
            r for index, r in enumerate(self.processed_form.tokenized_requests.entries)
            if index not in self.processed_form_errors and r.uid.data is not None
        ]

        if not entries:
            return dict()

        uids_to_secrets = dict()
        for t in entries:
            requested_secret_version = t.secret_version.data or ''
            if requested_secret_version:
                requested_secret_version = ulid.ULID(
                    requested_secret_version,
                    prefix=SecretVersionUUIDType.prefix,
                )

            tokenized_version = found_versions.get(
                (DelegationToken.hash_token(t.token.data), requested_secret_version)
            )
            if tokenized_version:
                uids_to_secrets.setdefault(t.uid.data, set()).add(tokenized_version['secret_uuid'])

        result = dict(
            map(
                lambda x: (x[0], self._fetch_readable_secrets_for_user(x[0], x[1])),
                uids_to_secrets.items(),
            )
        )
        return result

    def _fetch_readable_secrets_for_user(self, uid, secrets):
        query = union(
            select(
                [UserRole.secret_uuid.label('secret_uuid').label('secret_uuid')],
                and_(
                    UserRole.secret_uuid.in_(secrets),
                    UserRole.role_id.in_([Roles.READER.value, Roles.OWNER.value]),
                    UserRole.external_type == 'user',
                    UserRole.uid == uid,
                ),
            ),
            select(
                [UserRole.secret_uuid.label('secret_uuid').label('secret_uuid')],
                UserRole.secret_uuid.in_(secrets),
                join(
                    UserRole,
                    ExternalRecord,
                    and_(
                        UserRole.secret_uuid.in_(secrets),
                        UserRole.role_id.in_([Roles.READER.value, Roles.OWNER.value]),
                        UserRole.external_type == 'staff',
                        ExternalRecord.uid == uid,
                        UserRole.external_type == ExternalRecord.external_type,
                        UserRole.staff_id == ExternalRecord.external_id,
                    ),
                )
            ),
            select(
                [UserRole.secret_uuid.label('secret_uuid').label('secret_uuid')],
                UserRole.secret_uuid.in_(secrets),
                join(
                    UserRole,
                    ExternalRecord,
                    and_(
                        UserRole.secret_uuid.in_(secrets),
                        UserRole.role_id.in_([Roles.READER.value, Roles.OWNER.value]),
                        UserRole.external_type == 'abc',
                        ExternalRecord.uid == uid,
                        UserRole.external_type == ExternalRecord.external_type,
                        UserRole.abc_id == ExternalRecord.external_id,
                        UserRole.abc_scope_id == ExternalRecord.external_scope_id,
                        UserRole.abc_role_id == ExternalRecord.external_role_id,
                    ),
                )
            ),
        )

        result = set(
            map(
                lambda x: x['secret_uuid'],
                get_db().session.execute(query),
            ),
        )
        return result
