# -*- coding: utf-8 -*-

from base64 import b64decode

from passport.backend.utils.common import noneless_dict
from passport.backend.vault.api.errors import (
    EmptyDiffSecretVersionError,
    HeadVersionNotFoundError,
    NonexistentEntityError,
    SecretHasNewHeadVersionError,
    ValidationError,
)
from passport.backend.vault.api.models import (
    Bundle,
    BundleVersion,
    Roles,
    Secret,
    SecretVersion,
)
from passport.backend.vault.api.models.base import State
from passport.backend.vault.api.utils.errors import InvalidUUIDPrefix
from passport.backend.vault.api.views.base_view import BaseView
from passport.backend.vault.api.views.versions.forms import (
    CreateDiffSecretVersionForm,
    CreateSecretVersionForm,
    UpdateSecretVersionForm,
)
from sqlalchemy import desc


class BaseVersionView(BaseView):
    def validate_value_field(self, field, num=0, error_field_name=None):
        error_field_name = error_field_name or 'value'
        if field.value.data is not None and field.encoding.data == 'base64':
            try:
                b64decode(field.value.data.encode('utf-8'))
            except TypeError:
                raise ValidationError(
                    errors={
                        error_field_name: [
                            {'index': num, field.key.data: ['invalid_base64']}
                        ]
                    }
                )

    def collect_value(self, form_field, error_field_name=None):
        value = []
        for i, kv in enumerate(form_field):
            self.validate_value_field(kv, i, error_field_name)
            value.append(noneless_dict({
                'key': kv.key.data,
                'value': kv.value.data,
                'encoding': kv.encoding.data,
            }))
        return value

    def resolve_version(self, uuid, required_role=Roles.READER):
        entity_prefix, entity_uuid = uuid.lower().split('-', 2)
        if entity_prefix == 'sec':
            self.check_if_has_role(role=required_role, secret_uuid=entity_uuid)
            # Когда достаем по айди секрета, отсекаем заэкспайренные и скрытые версии
            version = SecretVersion.query.filter(
                SecretVersion.secret_uuid == entity_uuid,
                SecretVersion.only_visible_versions_filter(),
            ).order_by(
                desc(SecretVersion.version)
            ).limit(1).one_or_none()
            if not version:
                raise NonexistentEntityError(SecretVersion, 'HEAD')

        elif entity_prefix == 'ver':
            # Когда достаем по адйи версии ищем и сскрытые и заэкспайренные
            version = SecretVersion.get_by_id(entity_uuid)
            self.check_if_has_role(role=required_role, secret_uuid=version.secret_uuid)

        elif entity_prefix == 'bun':
            self.check_if_has_role(role=required_role, bundle_uuid=entity_uuid)
            bundle_versions = list(Bundle.get_by_id(entity_uuid).bundle_versions)
            if not bundle_versions:
                raise NonexistentEntityError(BundleVersion, 'HEAD')
            version = bundle_versions[0]

        elif entity_prefix == 'bve':
            version = BundleVersion.get_by_id(entity_uuid)
            self.check_if_has_role(role=required_role, bundle_uuid=version.bundle_uuid)

        else:
            raise InvalidUUIDPrefix(version_prefix=entity_prefix)

        self.statbox_log(
            action='get',
            version_prefix=entity_prefix,
            version_uuid=uuid,
            state=version.state_name(),
            expired_at=version.expired,
        )

        return entity_prefix, entity_uuid, version


class GetVersionView(BaseVersionView):
    """
    Получить версию секрета. В ручку можно передать айди секрета или версии.
    Для секрета вернет head-версию.
    -----
    returns = ['version']
    raises = [NonexistentEntityError]
    example = {
        'headers': {'Content-Type': 'application/json'},
        'arguments': {
            '<string:version_or_secret_uuid>': 'ver-0000000000000000000000nbk1',
        },
        'data': {},
        'response': {
            'version': {
                'created_at': 1445385600,
                'created_by': 100,
                'creator_login': 'vault-test-100',
                'secret_name': 'secret_1',
                'value': [
                    {
                        'key': 'password',
                        'value': '123',
                    },
                    {
                        'key': 'cert',
                        'value': 'U2VjcmV0IGZpbGU=',
                        'encoding': 'base64',
                    },
                ],
                'version': 'ver-0000000000000000000000nbk1',
                'secret_uuid': 'sec-0000000000000000000000ygj0',
            },
            'status': 'ok'
        }
    }
    """
    statbox_mode = 'version'
    required_user_auth = True
    use_slave = True

    def process_request(self, version_or_secret_uuid):
        version_prefix, version_uuid, version = self.resolve_version(version_or_secret_uuid, required_role=Roles.READER)

        self.check_state(version=version, response_dict=self.response_values)
        self.check_version_expiration(version, self.response_values)

        included_fields = ['secret_uuid', 'parent_version_uuid', 'updated_at', 'updated_by']
        if version.state != State.normal.value:
            included_fields.append('state_name')

        serialized_version = version.serialize(
            include=included_fields,
            exclude=['keys'],
        )

        self.response_values.update({
            'version': serialized_version,
        })


class UpdateVersionView(BaseVersionView):
    """
    Обновить версию секрета
    -----
    raises = [NonexistentEntityError]
    example = {
        'arguments': {
            '<string:version_uuid>': 'ver-0000000000000000000000ygj2',
        },
        'data': 'state=hidden',
        'response': {
            'status': 'ok',
        },
    }
    """

    form = UpdateSecretVersionForm
    statbox_mode = 'update_version'
    required_user_auth = True
    use_slave = False

    def process_request(self, version_uuid):
        version_prefix, version_uuid, version = self.resolve_version(uuid=version_uuid, required_role=Roles.OWNER)
        if self.processed_form.state.data is not None:
            version.state = State[self.processed_form.state.data].value
        if self.processed_form.comment.data is not None:
            version.comment = self.processed_form.comment.data
            if version.comment == '':
                version.comment = None
        version.set_ttl(self.processed_form.ttl.data)
        version.touch(self.validated_uid)
        self.commit(version)

        self.statbox_log(
            action='success',
            version_uuid=str(version.version),
            state=version.state_name(),
            expired_at=version.expired_at,
            comment=version.comment,
        )


class CreateDiffVersionView(BaseVersionView):
    """
    Создать новую версию секрета из диффа
    -----
    returns = ['secret_version']
    raises = [NonexistentEntityError]
    example = {
        'arguments': {'<string:secret_uuid>': 'sec-0000000000000000000000ygj0'},
        'headers': {'Content-Type': 'application/json'},
        'data': {
            'comment': 'new version',
            'diff': [{
                'key': 'login',
            }, {
                'key': 'password',
                'value': '123456',
            }, {
                'key': 'cert',
                'value': 'U2VjcmV0IGZpbGU=',
                'encoding': 'base64',
            }],
        },
        'response': {
            'secret_version': 'ver-0000000000000000000000ygj2',
            'status': 'ok',
        },
    }
    """
    form = CreateDiffSecretVersionForm
    statbox_mode = 'create_diff_version'

    def _pack_value(self, unpacked_value):
        value = {}
        for el in unpacked_value:
            value[el['key']] = el
        return value

    def patch_value(self, value, diff):
        packed = self._pack_value(value)
        diff = self.collect_value(diff, error_field_name='diff')
        for i, diff_el in enumerate(diff):
            if 'value' not in diff_el:
                if diff_el['key'] not in packed:
                    raise ValidationError(errors={'diff': [{'index': i, diff_el['key']: ['not_exists']}]})
                packed.pop(diff_el['key'])
            else:
                packed[diff_el['key']] = diff_el
        return packed.values()

    def check_head_version(self, parent_version_uuid, secret_uuid):
        if self.processed_form.check_head.data:
            secret_head_version = Secret.get_by_id(secret_uuid).head_version()
            if not secret_head_version:
                raise HeadVersionNotFoundError()

            if secret_head_version.version != parent_version_uuid:
                raise SecretHasNewHeadVersionError()

    def check_version_has_keys(self, secret_value):
        if not len(secret_value):
            raise EmptyDiffSecretVersionError()

    def process_request(self, parent_version_uuid):
        parent_secret_version = SecretVersion.get_by_id(parent_version_uuid)
        self.check_if_has_role(role=[Roles.OWNER, Roles.APPENDER], secret_uuid=parent_secret_version.secret_uuid)
        self.check_head_version(parent_version_uuid, parent_secret_version.secret_uuid)

        patched_value = self.patch_value(parent_secret_version.value, self.processed_form.diff.entries)
        self.check_version_has_keys(patched_value)

        new_secret_version = SecretVersion.create_secret_version(
            created_by=self.validated_uid,
            secret=parent_secret_version.secret,
            value=patched_value,
            comment=self.processed_form.comment.data,
            parent_version_uuid=parent_secret_version.version,
            ttl=self.processed_form.ttl.data,
        )
        self.commit(new_secret_version)

        self.response_values.update(
            new_secret_version.serialize(
                include=['secret_uuid', 'parent_version_uuid', 'parent_diff_keys'],
                exclude=['value', 'created_by', 'created_at', 'comment', 'keys', 'secret_name'],
            ),
        )

        self.statbox_log(
            action='success',
            version_uuid=str(new_secret_version.version),
            parent_version_uuid=str(parent_secret_version.version),
            expired_at=new_secret_version.expired_at,
        )


class CreateSecretVersionView(BaseVersionView):
    """
    Создать новую версию секрета
    -----
    returns = ['secret_version']
    raises = [NonexistentEntityError]
    example = {
        'arguments': {'<string:secret_uuid>': 'sec-0000000000000000000000ygj0'},
        'headers': {'Content-Type': 'application/json'},
        'data': {
            'value': [{
                'key': 'login',
                'value': 'ppodolsky',
            }, {
                'key': 'password',
                'value': '123456',
            }, {
                'key': 'cert',
                'value': 'U2VjcmV0IGZpbGU=',
                'encoding': 'base64',
            }],
        },
        'response': {
            'secret_version': 'ver-0000000000000000000000ygj2',
            'status': 'ok',
        },
    }
    """
    form = CreateSecretVersionForm
    statbox_mode = 'create_secret_version'

    def create_secret_version(self, secret, form):
        value = self.collect_value(form.value.entries)
        secret_version = SecretVersion.create_secret_version(
            secret=secret,
            created_by=self.validated_uid,
            value=value,
            comment=form.comment.data,
            ttl=form.ttl.data,
        )
        self.add(secret_version)
        self.response_values.update({
            'secret_version': secret_version.version.str(),
        })
        return secret_version

    def process_request(self, secret_uuid):
        self.check_if_has_role(role=[Roles.OWNER, Roles.APPENDER], secret_uuid=secret_uuid)
        secret = Secret.get_by_id(secret_uuid)
        version = self.create_secret_version(secret=secret, form=self.processed_form)
        self.commit()

        self.statbox_log(
            action='success',
            version_uuid=str(version.version),
            expired_at=version.expired_at,
        )
