# coding: utf-8

import argparse
from collections import (
    defaultdict,
    OrderedDict,
)
import contextlib
import json
import sys
import textwrap
import traceback

import six


class CLIArgumentError(ValueError):
    pass


class Namespace(argparse.Namespace):
    def __getattr__(self, name):
        if name.startswith('_'):
            raise AttributeError('\'{}\' object has no attribute \'{}'.format(self.__class__.__name__, name))
        return None


class JSONArgument(argparse.Action):
    def __call__(self, parser, namespace, values, option_string=None):
        try:
            values = json.loads(values)
        except ValueError as e:
            if option_string:
                raise CLIArgumentError('{}: {}'.format(option_string, str(e)))
            raise  # pragma: no cover
        setattr(namespace, self.dest, values)


class CLICommand(object):
    def __init__(self, args_parser, stdout=None, stderr=None, output_encoding='utf-8'):
        self.args_parser = args_parser
        self._distinct_groups = defaultdict(
            self.args_parser.add_mutually_exclusive_group,
        )
        self.args_parser.description = textwrap.dedent(self.__doc__) if self.__doc__ else None
        self.stdout = stdout
        self.stderr = stderr
        self.output_encoding = output_encoding

    @property
    def usage(self):
        return self.args_parser.usage

    @usage.setter
    def usage(self, value):
        self.args_parser.usage = textwrap.dedent(value).strip()

    def add_argument(self, *args, **kwargs):
        distinct_group = kwargs.pop('distinct_group', None)
        parser = self.args_parser
        if distinct_group:
            parser = self._distinct_groups[distinct_group]
        parser.add_argument(*args, **kwargs)

    def _get_rsa_auth_from_args(self, args):
        try:
            from library.python.vault_client.auth import (
                RSASSHAgentAuth,
                RSASSHAgentHash,
                RSAPrivateKeyAuth,
            )
        except ImportError:
            from vault_client.auth import (
                RSASSHAgentAuth,
                RSASSHAgentHash,
                RSAPrivateKeyAuth,
            )

        rsa_auth = None
        rsa_login = args.rsa_login

        if args.rsa_private_key_file is not None:
            rsa_auth = RSAPrivateKeyAuth(args.rsa_private_key_file.read())
            args.rsa_private_key_file.close()
            if not rsa_login:
                self.args_parser.error('--rsa-private-key requires --rsa-login')
        elif args.rsa_agent_key_num is not None:
            rsa_auth = RSASSHAgentAuth(key_num=args.rsa_agent_key_num)
        elif args.rsa_agent_key_hash is not None:
            rsa_auth = RSASSHAgentHash(args.rsa_agent_key_hash)
        else:
            rsa_auth = RSASSHAgentAuth()

        return rsa_login, rsa_auth

    def echo(self, line='', nl=True, err=False):
        end = six.b('\n' if nl else '')
        if err:
            out = self.stderr or sys.stderr
        else:
            out = self.stdout or sys.stdout

        if six.PY3:
            out = out.buffer

        if isinstance(line, six.text_type):
            out.write(line.encode(self.output_encoding))
            out.write(end)
        elif isinstance(line, six.binary_type):
            out.write(line)
            out.write(end)
        else:
            six.print_(line, end=end, file=out)

    def exit(self, code=0):
        sys.exit(code)

    def process_args(self, cli_args):
        return cli_args

    def process(self, cli_args, *args, **kwargs):
        self.echo('Process the command')  # pragma: no cover


class CLIArgumentParser(argparse.ArgumentParser):
    def error(self, message, stderr=None):
        stderr = stderr or sys.stderr
        six.print_('error: %s' % message, file=stderr)
        six.print_('-' * 20, file=stderr)
        self.print_help(file=stderr)
        sys.exit(2)


class CLIManagerCommandRecord(defaultdict):
    def __init__(self, **kwargs):
        super(CLIManagerCommandRecord, self).__init__(lambda: None, **kwargs)
        self.__dict__ = self


class CLIManager(CLICommand):
    def __init__(self, doc=None, stdout=None, stderr=None, debug=False):
        super(CLIManager, self).__init__(
            args_parser=CLIArgumentParser(
                formatter_class=argparse.RawDescriptionHelpFormatter,
            ),
            stdout=stdout,
            stderr=stderr,
        )
        self.debug = debug
        self._root_subparsers = self.args_parser.add_subparsers(dest='_command_1', metavar='command')
        self._entities_subparsers = dict()

        self._commands = OrderedDict()

        if doc:
            self.args_parser.description = doc

    def assign_command(self, name, class_, help=None, *args, **kwargs):
        command = CLIManagerCommandRecord(
            name=name,
            class_=class_,
            args=args,
            kwargs=kwargs,
            help=help,
            object=None,
            parser=self._get_parser(name),
        )
        command.object = class_(command['parser'], stdout=self.stdout, stderr=self.stderr, *args, **kwargs)
        self._commands[name] = command

    def print_help(self, message=None, err=False, args=None):
        if message:
            self.echo(message, err=err)
            self.echo('-' * 20, err=err)

        command = self.get_command(' '.join(args[0:2]).strip()) if args else ''
        args_parser = command.args_parser if command else self.args_parser

        args_parser.print_help(
            file=sys.stderr if err else sys.stdout,
        )

    def get_command(self, name):
        if name in self._commands:
            return self._commands[name].object
        else:  # pragma: no cover
            return None

    def parse_args(self, args, **kwargs):
        self.args_parser.epilog = (
            'available commands:\n' +
            '\n'.join('  ' + c for c in self._commands.keys())
        )

        self.args_parser.epilog += '\n\nRead the documentation on the website https://vault-api.passport.yandex.net/docs/'

        for ep in self._entities_subparsers.values():
            ep['parser'].epilog = (ep['parser'].epilog or '') + self.args_parser.epilog
            ep['parser'].formatter_class = argparse.RawDescriptionHelpFormatter
            if 'subparser' in ep:
                for sp in ep['subparser']._name_parser_map.values():
                    if isinstance(sp, CLIArgumentParser):
                        sp.epilog = (sp.epilog or '') + self.args_parser.epilog
                        sp.formatter_class = argparse.RawDescriptionHelpFormatter

        with self.process_errors(exit_code=2, print_help=True, args=args):
            return self.args_parser.parse_args(args, **kwargs)

    @contextlib.contextmanager
    def process_errors(self, exit_code=3, print_help=False, args=None):
        try:
            yield
        except Exception as e:
            if isinstance(e, CLIArgumentError):
                exit_code = 2
                print_help = True

            if self.debug:
                traceback.print_exc(file=self.stderr)

            e_kwargs = getattr(e, 'kwargs', {})
            message = 'error: ' + (e_kwargs.get('message') or str(e))
            errors = e_kwargs.get('errors')
            if errors:
                message += ' (' + str(errors) + ')'

            if print_help:
                self.print_help(message, args=args, err=True)
            else:
                self.echo(message, err=True)

            self.exit(exit_code)

    def process(self, args=None, commands_kwargs=None):
        parsed_args = self.parse_args(args, namespace=Namespace())

        if '_command_2' in parsed_args:
            name = ' '.join([
                parsed_args._command_1 or '',
                parsed_args._command_2 or '',
            ])
        else:
            name = parsed_args._command_1 or ''
        command = self.get_command(name)
        if command is not None:
            with self.process_errors(args=args):
                command.process(command.process_args(parsed_args), **(commands_kwargs or {}))
        else:
            self.print_help()  # pragma: no cover

    def _get_parser(self, name):
        parts = name.split(' ')[:2]

        if parts[0] not in self._entities_subparsers:
            parser = dict(
                parser=self._root_subparsers.add_parser(parts[0]),
            )
            if len(parts) > 1:
                parser['subparser'] = parser['parser'].add_subparsers(
                    dest='_command_2',
                    metavar='command',
                )
            self._entities_subparsers[parts[0]] = parser
        else:
            parser = self._entities_subparsers[parts[0]]  # pragma: no cover

        if len(parts) > 1:
            parser = parser['subparser'].add_parser(parts[1])
        else:
            parser = parser['parser']
        return parser
