# coding: utf-8

from collections import namedtuple


try:
    from library.python.vault_client.client import VaultClient as OriginalVaultClient
    from library.python.vault_client.instances import (
        VAULT_PRODUCTION_API as ORIGINAL_VAULT_PRODUCTION_API,
        VAULT_TESTING_API as ORIGINAL_VAULT_TESTING_API,
    )
    from library.python.vault_client.utils import noneless_dict
except ImportError:
    from vault_client.client import VaultClient as OriginalVaultClient
    from vault_client.instances import (
        VAULT_PRODUCTION_API as ORIGINAL_VAULT_PRODUCTION_API,
        VAULT_TESTING_API as ORIGINAL_VAULT_TESTING_API,
    )
    from vault_client.utils import noneless_dict

VAULT_PRODUCTION_API = ORIGINAL_VAULT_PRODUCTION_API
VAULT_TESTING_API = ORIGINAL_VAULT_TESTING_API


class VaultClient(OriginalVaultClient):
    def suggest(self, query, limit=None, return_raw=False):
        """Search objects by query.

        Args:
            query: Query
            return_raw: If set to True raw response will be returned instead of processed one

        Returns:
            Lists of objects from Staff and ABC
            or raw response depending on return_raw parameter.

        Raises:
            ClientError: An error occurred doing request
        """
        url = self.host + '/suggest/'

        params = noneless_dict({
            'query': query.encode('utf-8'),
            'limit': limit,
        })

        r = self._call_native_client('get', url, params=params, skip_auth=True)
        if return_raw:
            return r  # pragma: no cover

        data = self.validate_response(r)
        return data


CLIVaultClientResponse = namedtuple('CLIVaultClient', ['result', 'success', 'e'])


class CLIVaultClient(object):
    def __init__(self, vault_client, rsa_auth=None):
        self.vault_client = vault_client
        self.rsa_auth = rsa_auth

        try:
            from library.python.vault_client.errors import ClientError
        except ImportError:
            from vault_client.errors import ClientError

        self.ClientError = ClientError

    def _call_client(self, method_name, *args, **kwargs):
        kwargs = dict(kwargs)
        skip_default_kwargs = kwargs.pop('skip_default_kwargs', False)
        if self.rsa_auth:
            kwargs['rsa_auth'] = self.rsa_auth
        if not skip_default_kwargs:
            pass
        return getattr(self.vault_client, method_name)(*args, **kwargs)

    def get_abc_id_by_name(self, name):
        if not name:
            return
        suggest = self._call_client(
            'suggest',
            query=name,
        )
        for service in suggest.get('abc_departments', []):
            if service['unique_name'] == name:
                return str(service['id'])

    def get_staff_id_by_name(self, name):
        if not name:
            return
        suggest = self._call_client(
            'suggest',
            query=name,
        )
        for service in suggest.get('staff_departments', []):
            if service['unique_name'] == name:
                return str(service['id'])

    def resolve_services(self, role):
        abc_id = role.get('abc_id')
        if abc_id is not None and not abc_id.isdigit():
            abc_id = self.get_abc_id_by_name(abc_id)
            if abc_id is None:
                raise self.ClientError(
                    message='"{}" is an unknown ABC service name'.format(role.get('abc_id')),
                )
            role['abc_id'] = abc_id

        staff_id = role.get('staff_id')
        if staff_id is not None and not staff_id.isdigit():
            staff_id = self.get_staff_id_by_name(staff_id)
            if staff_id is None:
                raise self.ClientError(
                    message='"{}" is an unknown Staff service name'.format(role.get('staff_id')),
                )
            role['staff_id'] = staff_id

        return role

    def _update_secrets_roles(self, secret_uuid, roles):
        methods = {
            'assign': 'add_user_role_to_secret',
            'revoke': 'delete_user_role_from_secret',
        }
        for role in roles:
            self._call_client(
                methods[role.action],
                secret_uuid,
                role.role_type,
                **self.resolve_services(
                    role.as_method_params(),
                )
            )

    def _wrap_client_response(func):
        def wrapper(*args, **kwargs):
            try:
                from library.python.vault_client.errors import ClientError
            except ImportError:
                from vault_client.errors import ClientError

            try:
                result = func(*args, **kwargs)
                return CLIVaultClientResponse(result, True, None)
            except ClientError as e:
                return CLIVaultClientResponse(e.kwargs, False, e)
        return wrapper

    def _unpack_value(self, value):
        if isinstance(value, (tuple, list)):
            return value
        if isinstance(value, dict):
            real_value = []
            for k, v in value.items():
                real_value.append({'key': k, 'value': v})
            return real_value
        raise self.ClientError(message='Unknown value format')

    @_wrap_client_response
    def list_secrets(self, query=None, query_type=None, tags=None, page=None):
        return self._call_client(
            'list_secrets',
            query=query,
            query_type=query_type,
            tags=tags,
            page=page,
        )

    @_wrap_client_response
    def list_tokens(self, secret_uuid, with_revoked=False, page=None):
        return self._call_client(
            'list_tokens',
            secret_uuid,
            with_revoked=with_revoked,
            page=page,
        )

    @_wrap_client_response
    def create_secret(self, name, comment=None, roles=None, value=None, tags=None, version_ttl=None):
        secret_uuid = self._call_client(
            'create_secret',
            name=name,
            comment=comment,
            tags=tags,
        )
        if value:
            self._call_client(
                'create_secret_version',
                secret_uuid=secret_uuid,
                value=value,
                ttl=version_ttl,
            )

        if roles:
            self._update_secrets_roles(secret_uuid, roles)

        return self._call_client(
            'get_secret',
            secret_uuid,
        )

    @_wrap_client_response
    def update_secret(self, secret_uuid, name=None, update=False, comment=None, roles=None, value=None,
                      tags=None, version_ttl=None, state=None):
        if any(map(lambda x: x is not None, [name, comment, tags, state])):
            self._call_client(
                'update_secret',
                secret_uuid,
                name=name,
                comment=comment,
                tags=tags,
                state=state,
            )

        if value:
            old_version = None
            if update:
                # Fetch a HEAD version by a secret uuid
                try:
                    old_version = self._call_client(
                        'get_version',
                        secret_uuid,
                    )
                except self.ClientError as e:
                    if e.kwargs['code'] != 'nonexistent_entity_error':
                        raise

            if old_version:
                self._call_client(
                    'create_diff_version',
                    old_version['version'],
                    comment=comment,
                    diff=self._unpack_value(value),
                    ttl=version_ttl,
                )
            else:
                self._call_client(
                    'create_secret_version',
                    secret_uuid=secret_uuid,
                    comment=comment,
                    value=value,
                    ttl=version_ttl,
                )

        if roles:
            self._update_secrets_roles(secret_uuid, roles)

        return self._call_client(
            'get_secret',
            secret_uuid,
        )

    @_wrap_client_response
    def create_version(self, secret_uuid, value, update=False, roles=None, comment=None,
                       ttl=None, packed_value=True, old_version=None):
        if update:
            # Fetch a HEAD version by a secret uuid
            try:
                old_version = self._call_client(
                    'get_version',
                    secret_uuid,
                )
            except self.ClientError as e:
                if e.kwargs['code'] != 'nonexistent_entity_error':
                    raise

        if old_version:
            version_uuid = self._call_client(
                'create_diff_version',
                old_version['version'],
                comment=comment,
                diff=self._unpack_value(value),
                ttl=ttl,
            )
        else:
            version_uuid = self._call_client(
                'create_secret_version',
                secret_uuid=secret_uuid,
                comment=comment,
                value=value,
                ttl=ttl,
            )

        result = None
        try:
            if roles:
                self._update_secrets_roles(secret_uuid, roles)

            result = self._call_client(
                'get_version',
                version_uuid,
                packed_value=packed_value,
            )
        except self.ClientError as e:
            if e.kwargs['code'] != 'access_error':
                raise

        return result or version_uuid

    @_wrap_client_response
    def update_version(self, version_uuid, state=None, ttl=None, comment=None, packed_value=True):
        self._call_client(
            'update_version',
            version_uuid,
            state=state,
            ttl=ttl,
            comment=comment,
        )

        return self._call_client(
            'get_version',
            version_uuid,
            packed_value=packed_value,
        )

    @_wrap_client_response
    def create_token(self, secret_uuid, tvm_client_id=None, signature=None, comment=None):
        token, token_uuid = self._call_client(
            'create_token',
            secret_uuid,
            tvm_client_id=tvm_client_id,
            signature=signature,
            comment=comment,
        )
        return dict(
            secret_uuid=secret_uuid,
            token=token,
            token_uuid=token_uuid,
        )

    @_wrap_client_response
    def get_secret(self, secret_uuid):
        return self._call_client(
            'get_secret',
            secret_uuid,
        )

    @_wrap_client_response
    def get_version(self, uuid, packed_value=True):
        return self._call_client(
            'get_version',
            uuid,
            packed_value=packed_value,
        )

    @_wrap_client_response
    def get_token_info(self, token=None, token_uuid=None):
        return self._call_client(
            'get_token_info',
            token=token,
            token_uuid=token_uuid,
        )

    @_wrap_client_response
    def revoke_token(self, token_uuid):
        self._call_client(
            'revoke_token',
            token_uuid=token_uuid,
        )
        return self._call_client(
            'get_token_info',
            token_uuid=token_uuid,
        )

    @_wrap_client_response
    def restore_token(self, token_uuid):
        self._call_client(
            'restore_token',
            token_uuid=token_uuid,
        )
        return self._call_client(
            'get_token_info',
            token_uuid=token_uuid,
        )
