# coding: utf-8

import argparse
import base64
import os

from passport.backend.vault.utils.roles import RoleAction
from vault_client_cli.cli_base import JSONArgument


class RolesArgument(argparse.Action):
    def __call__(self, parser, namespace, values, option_string=None):
        result = getattr(namespace, self.dest, None) or []
        result += map(RoleAction.from_string, values)
        setattr(namespace, self.dest, result)


class KeyValueArgument(argparse.Action):
    def __call__(self, parser, namespace, values, option_string=None):
        result = getattr(namespace, self.dest, None) or {}
        for kv in values:
            if kv.find('=') < 1:
                raise ValueError('Bad value: {}'.format(kv))
            key, value = kv.split('=', 1)
            if value.strip() == '':
                raise ValueError('The value can not be empty or consist only of whitespaces: {}'.format(kv))
            result[key] = value
        setattr(namespace, self.dest, result)


class KeyValueFileArgument(argparse.Action):
    def __call__(self, parser, namespace, values, option_string=None):
        result = getattr(namespace, self.dest, None) or []
        for kv in values:
            if kv.find('=') >= 0:
                key, value = kv.split('=', 1)
            else:
                key, value = '', kv

            if value.strip() == '':
                raise ValueError('The filename can not be empty or consist only of whitespaces: {}'.format(kv))
            result.append(self._read_file(key, value))
        setattr(namespace, self.dest, result)

    def _read_file(self, key, filename):
        if not os.path.exists(filename):
            raise ValueError('File "{}" not found'.format(filename))
        with open(filename, 'rb') as f:
            result = dict(
                key=key if key else os.path.basename(filename),
                value=base64.b64encode(f.read()).decode('latin-1'),
                encoding='base64',
            )
        return result


CLI_VAULT_BASE_ARGUMENTS = {
    'testing': {
        'args': ('-T', '--testing'),
        'kwargs': dict(
            dest='testing',
            action='store_true',
            help='Send requests to a testing environment',
        ),
    },

    'json': {
        'args': ('-j', '--json'),
        'kwargs': dict(
            dest='as_json',
            action='store_true',
            default=False,
            help='Print result as JSON',
        ),
    },

    'compact': {
        'args': ('--compact',),
        'kwargs': dict(
            dest='compact',
            action='store_true',
            default=False,
            help='Print result in a compact form',
        ),
    },

    'rsa_agent_key_num': {
        'args': ('--rsa-agent-key-num',),
        'kwargs': dict(
            distinct_group='rsa_key',
            dest='rsa_agent_key_num',
            metavar='KEY_NUMBER',
            action='store',
            type=int,
            help='''
                Key number in ssh-agent. Key numbers start from zero.
                You can get the list of keys in the agent by the command 'ssh-add -l'
            ''',
        ),
    },

    'rsa_agent_key_hash': {
        'args': ('--rsa-agent-key-hash',),
        'kwargs': dict(
            distinct_group='rsa_key',
            dest='rsa_agent_key_hash',
            metavar='KEY_HASH',
            help='''
                SSH key hash from the ssh agent. You can get the list of keys in the agent by the command 'ssh-add -l'
            ''',
        ),
    },

    'rsa_private_key_file': {
        'args': ('--rsa-private-key',),
        'kwargs': dict(
            distinct_group='rsa_key',
            dest='rsa_private_key_file',
            metavar='FILENAME',
            type=argparse.FileType('rt'),
            help='RSA private key filename.',
        ),
    },

    'rsa_login': {
        'args': ('--rsa-login',),
        'kwargs': dict(
            dest='rsa_login',
            metavar='login',
            help='RSA private key login.',
            default=None,
        ),
    },

    'oauth_token': {
        'args': ('--oauth',),
        'kwargs': dict(
            distinct_group='rsa_key',
            dest='oauth_token',
            metavar='token',
            help='OAuth token. Alternative: put the token in the YAV_TOKEN environment variable. Get token via "yav oauth" command or at https://nda.ya.ru/3UVsAN',
            default=None,
        ),
    },

    'secret_comment': {
        'args': ('-c', '--comment',),
        'kwargs': dict(dest='comment', metavar='<comment>', help='A secret comment'),
    },

    'version_comment': {
        'args': ('-c', '--comment',),
        'kwargs': dict(dest='comment', metavar='<comment>', help='A version comment'),
    },

    'token_comment': {
        'args': ('-c', '--comment',),
        'kwargs': dict(dest='comment', metavar='<comment>', help='A token comment'),
    },

    'secret_uuid': {
        'args': ('secret_uuid',),
        'kwargs': dict(help='Secret UUID'),
    },

    'token_uuid': {
        'args': ('token_uuid',),
        'kwargs': dict(help='Token UUID'),
    },

    'secret_name': {
        'args': ('-n', '--name',),
        'kwargs': dict(dest='name', metavar='<name>', help='Non unique secret name'),
    },

    'secret_state': {
        'args': ('--state', ),
        'kwargs': dict(
            help='Secret state',
            dest='secret_state',
            choices=['hidden', 'normal'],
        ),
    },

    'uuid': {
        'args': ('uuid',),
        'kwargs': dict(metavar='version_uuid', help='Any of uids: a secret, a version, a bundle'),
    },

    'version_uuid': {
        'args': ('version_uuid',),
        'kwargs': dict(help='Version UUID'),
    },

    'version_state': {
        'args': ('--state', ),
        'kwargs': dict(
            help='Version state',
            dest='version_state',
            choices=['hidden', 'normal'],
        ),
    },

    'version_ttl': {
        'args': ('--ttl', ),
        'kwargs': dict(
            dest='version_ttl',
            type=int,
            help='Version ttl in seconds. 0 — reset ttl.',
            metavar='SECONDS',
        ),
    },

    'roles': {
        'args': ('-r', '--roles',),
        'kwargs': dict(
            dest='roles',
            nargs='+',
            metavar='<role>',
            action=RolesArgument,
            help='''
                Role description.
                Format: [+-](reader|owner):(user|abc|staff):(id|slug)[:(scope:abc_scope|role:abc_role_id)]
                Examples: owner:user:arhibot, -reader:staff:yandex, +reader:abc:passp, +reader:abc:123:scope:administration, -reader:abc:123:role:630
            ''',
        ),
    },

    'tags': {
        'args': ('-t', '--tags',),
        'kwargs': dict(dest='tags', metavar='<tags>', help='Comma-separated list of tags'),
    },

    'query': {
        'args': ('-q', '--query',),
        'kwargs': dict(dest='query', metavar='<query>', help='Search query string'),
    },

    'query_type': {
        'args': ('--query-type',),
        'kwargs': dict(
            dest='query_type',
            choices=['infix', 'language', 'exact', 'prefix'],
            metavar='<query_type>',
            help='Search query type. Choose from: infix, language, exact, prefix.',
        ),
    },

    'token_signature': {
        'args': ('-s', '--signature',),
        'kwargs': dict(dest='signature', metavar='signature', help='A token signature'),
    },

    'token_tvm_client_id': {
        'args': ('-tvm', '--tvm-client-id',),
        'kwargs': dict(
            dest='tvm_client_id',
            metavar='tvm-client-id',
            type=int,
            help='A TVM Client ID binded to token',
        ),
    },

    'update_value': {
        'args': ('-u', '--update',),
        'kwargs': dict(
            dest='update',
            action='store_true',
            help='Create a new secrets version from last secret version',
        ),
    },

    'value': {
        'args': ('-v', '--value',),
        'kwargs': dict(
            distinct_group='value',
            dest='value',
            action=JSONArgument,
            metavar='<json>',
            help='''
                Secret value, can be in packed format
                (i.e. {"username": "ppodolsky", "password": "123456"})
                or in native format
                (i.e. [{"key": "username", "value": "ppodolsky"}, {"key": "password", "value": "123456"}])
            ''',
        ),
    },

    'kv': {
        'args': ('-k', '--key-value',),
        'kwargs': dict(
            distinct_group='value',
            dest='kv',
            action=KeyValueArgument,
            nargs='+',
            metavar='<key=value>',
            help='The key and value for the version',
        ),
    },

    'files_kv': {
        'args': ('-f', '--key-filename',),
        'kwargs': dict(
            dest='files_kv',
            action=KeyValueFileArgument,
            nargs='+',
            metavar='<key=filename>',
            help='The key and filename for the version. The contents of the file will be stored in a secret in base64 format',
        ),
    },

    'delete_keys': {
        'args': ('-d', '--delete-key',),
        'kwargs': dict(
            dest='delete_keys',
            nargs='+',
            metavar='<key>',
            help='Keys that do not need to be copied from the old version. Use only with the -u option',
        ),
    },

    'only_value': {
        'args': ('-o', '-O', '--only-value'),
        'kwargs': dict(
            dest='only_value',
            nargs='?',
            default='',
            help='Show a secret value only. If you specify key_name, then get only the value of the key.',
            metavar='KEY_NAME',
        ),
    },

    'skip_nl': {
        'args': ('-n',),
        'kwargs': dict(
            dest='skip_nl',
            action='store_true',
            default=False,
            help='do not print a new line at the end if get only the value with the key parameter is shown',
        ),
    },

    'value_format': {
        'args': ('--format',),
        'kwargs': dict(
            dest='value_format',
            choices=['json', 'java', 'yaml'],
            help='Format value as json (default) or java properties format or YAML',
        ),
    },

    'page': {
        'args': ('-p', '--page',),
        'kwargs': dict(dest='page', type=int),
    },

    'with_revoked': {
        'args': ('--with-revoked',),
        'kwargs': dict(
            dest='with_revoked',
            action='store_true',
            help='Add revoked tokens to the list',
        ),
    },
}
