# coding: utf-8

import json
import textwrap

from vault_client_cli.cli_base import (
    CLIArgumentError,
    CLICommand,
)

from .args import CLI_VAULT_BASE_ARGUMENTS
from .format import (
    format_comment,
    format_creator,
    format_role,
    format_table,
    format_timestamp,
    format_tokens,
    value_to_java_properties,
    value_to_yaml,
)


CLI_SUCCESS_EXIT_CODE = 0
CLI_ERROR_EXIT_CODE = 1

JSON_DUMPS_DEFAULT_ARGS = dict(ensure_ascii=True, indent=4, sort_keys=True)
JSON_DUMPS_HUMANS_ARGS = dict(JSON_DUMPS_DEFAULT_ARGS, ensure_ascii=False)


def merge_dicts(d1, *args):
    merged = dict(d1 or {})
    for d in args:
        if d is not None:
            merged.update(d)
    return merged


class VaultCLIBaseCommand(CLICommand):
    def __init__(self, *args, **kwargs):
        super(VaultCLIBaseCommand, self).__init__(*args, **kwargs)
        self.common_args()
        self.auth_args()

    def common_args(self):
        self.add_argument('--skip-warnings', action='store_true', dest='skip_warnings', help='skip warnings')

    def auth_args(self):
        self.add_base_argument('rsa_agent_key_num')
        self.add_base_argument('rsa_agent_key_hash')
        self.add_base_argument('rsa_private_key_file')
        self.add_base_argument('rsa_login')
        self.add_base_argument('oauth_token')

    def add_base_argument(self, name):
        base = CLI_VAULT_BASE_ARGUMENTS[name]
        self.add_argument(*base['args'], **base['kwargs'])


class VaultCLICommand(VaultCLIBaseCommand):
    def __init__(self, *args, **kwargs):
        super(VaultCLICommand, self).__init__(*args, **kwargs)
        self.as_json = False
        self.compact = False
        self.value_format = 'json'

        self.add_base_argument('testing')
        self.add_base_argument('json')
        self.add_base_argument('compact')

    def process_args(self, cli_args):
        self.as_json = cli_args.as_json
        self.compact = cli_args.compact

        return cli_args

    def _pack_value(self, unpacked_value):
        value = {}
        for el in unpacked_value:
            value[el['key']] = el['value']
        return value

    def get_value(self, cli_args, skip_exception=False):
        result = merge_dicts(
            self._pack_value(cli_args.value) if isinstance(cli_args.value, (list, tuple,)) else cli_args.value,
            cli_args.kv,
        )
        result = [{'key': k, 'value': v} for k, v in result.items()]

        if cli_args.files_kv:
            result.extend(cli_args.files_kv)

        if cli_args.update and cli_args.delete_keys:
            result.extend([dict(key=k) for k in cli_args.delete_keys])

        if not result and not skip_exception:
            raise CLIArgumentError('No version value specified')

        return result

    def process(self, cli_args, client, debug=False, *args, **kwargs):  # pragma: no cover
        self.echo('Process ' + self.__class__.__name__ + '...')
        self.echo(cli_args)

    def serialize_response(self, response):
        if self.as_json:
            if response.success:
                return self.serialize_success_response_as_json(response)
            else:
                return self.serialize_error_response_as_json(response)

        if response.success:
            return self.serialize_success_response(response)
        else:
            return self.serialize_error_response(response)

    def serialize_success_response(self, response):  # pragma: no cover
        return json.dumps(response.result, **JSON_DUMPS_HUMANS_ARGS) + '\n'

    def serialize_success_response_as_json(self, response):
        return json.dumps(response.result, **JSON_DUMPS_DEFAULT_ARGS) + '\n'

    def serialize_error_response(self, response):
        code = response.result.get('code', 'unknown_error')
        message = response.result.get('message')
        errors = response.result.get('errors')
        if errors:
            message += ' (' + str(errors) + ')'
        return ('error: ' + (message or code)).rstrip('\n') + '\n'

    def serialize_error_response_as_json(self, response):
        return json.dumps(response.result, **JSON_DUMPS_DEFAULT_ARGS) + '\n'

    def print_response_and_exit(self, response,
                                succes_exit_code=CLI_SUCCESS_EXIT_CODE,
                                error_exit_code=CLI_ERROR_EXIT_CODE):
        serialized_response = self.serialize_response(response)
        self.echo(serialized_response, err=not response.success, nl=False)
        self.exit(succes_exit_code if response.success else error_exit_code)

    def serialize_secret_response(self, response):
        return textwrap.dedent(u'''
            {state}
            uuid: {uuid}
            name: {name}
            comment: {comment}
            tags: {tags}

            creator: {creator}
            created: {created_at}
            updated: {updated_at}

            versions:
            {versions}

            roles:
            {roles}
            {tokens}
        ''').format(
            state='state: {}'.format(response.result.get('state_name', '')) if 'state_name' in response.result else '',
            uuid=response.result.get('uuid'),
            name=response.result.get('name', ''),
            versions=format_table(
                [
                    (
                        row['version'],
                        row.get('parent_version_uuid', '-'),
                        format_timestamp(row['created_at']),
                        format_creator(row),
                    )
                    for row
                    in response.result.get('secret_versions')
                ],
                header=['uuid', 'parent', 'created', 'creator'],
                compact=self.compact,
            ).rstrip() if response.result.get('secret_versions', False) else 'The secret hasn\'t versions.',
            roles='\n'.join(
                (format_role(row) for row in response.result['secret_roles']),
            ),
            created_at=format_timestamp(response.result.get('created_at')),
            updated_at=format_timestamp(response.result.get('updated_at')),
            tags=', '.join(response.result.get('tags', [])),
            comment=format_comment(response.result.get('comment', '')),
            creator=format_creator(response.result),
            tokens='\ntokens:\n' + format_tokens(response.result.get('tokens'), compact=self.compact) if response.result.get('tokens') else '',
        ).lstrip()

    def serialize_version_response(self, response):
        if self.value_format == 'java':
            value = value_to_java_properties(response.result.get('value'))
        elif self.value_format == 'yaml':
            value = value_to_yaml(response.result.get('value'))
        else:
            value = json.dumps(response.result.get('value'), **JSON_DUMPS_HUMANS_ARGS)
        return textwrap.dedent(u'''
            {state}
            version: {version_uuid}
            comment: {comment}
            secret uuid: {secret_uuid}
            secret name: {secret_name}
            parent: {parent}

            creator: {creator}
            created: {created_at}
            {expired_at}
            value:
            {value}
        ''').format(
            state='state: {}'.format(response.result.get('state_name', '')) if 'state_name' in response.result else '',
            secret_uuid=response.result.get('secret_uuid'),
            secret_name=response.result.get('secret_name', ''),
            version_uuid=response.result.get('version'),
            creator=format_creator(response.result),
            created_at=format_timestamp(response.result.get('created_at')),
            value=value,
            comment=format_comment(response.result.get('comment', '')),
            parent=response.result.get('parent_version_uuid', '-'),
            expired_at='expired: {}{}\n'.format(
                format_timestamp(response.result.get('expired_at')),
                ' (expired)' if response.result.get('expired', False) else '',
            ) if 'expired_at' in response.result else '',
        ).lstrip()
