# coding: utf-8

from datetime import datetime
import re
import textwrap

from passport.backend.utils.string import smart_unicode
from passport.backend.vault.utils.roles import format_role  # noqa
import six
from tabulate import tabulate
import yaml


DEFAULT_COMMENT_WIDTH = 80

# Fancy-grid разваливается под windows и не-utf консолями.
# Используем только ascii-символы для форматирования таблиц.
DEFAULT_TABULATE_STYLE = 'psql'
COMPACT_TABULATE_STYLE = 'simple'

# Эскейпим символы комментариев и разделители для java-properties
# В «ключе» дополнительно эскейпим пробелы и табуляцию
# https://en.wikipedia.org/wiki/.properties (VAULT-165)
ESCAPE_JAVA_PROP_RE = re.compile(r'([ \n:=!#\t])')
ESCAPE_JAVA_VALUE_RE = re.compile(r'([\n!#])')


def format_comment(comment, width=DEFAULT_COMMENT_WIDTH):
    prefix = ''
    if len(comment) > width:
        prefix = '\n'
    return prefix + '\n'.join(textwrap.wrap(comment, width)) + prefix


def format_table(table, header=None, compact=False):
    if not header:
        header = []
    if not header and not table:
        return ''
    return tabulate(
        table,
        headers=header,
        tablefmt={False: DEFAULT_TABULATE_STYLE}.get(compact, COMPACT_TABULATE_STYLE),
    ) + '\n'


def format_timestamp(ts):
    if not ts:
        return ''
    dt = datetime.fromtimestamp(ts)
    return dt.strftime('%Y-%m-%d %H:%M:%S')


def format_creator(data, skip_uid=False):
    created_by = data.get('created_by', None)
    creator_login = data.get('creator_login', None)
    if creator_login:
        return creator_login + (' ({})'.format(created_by) if not skip_uid else '')
    elif created_by:
        return six.text_type(created_by)
    else:
        return '—'


def format_tokens(tokens, compact=False):
    return format_table(
        [
            (
                format_timestamp(row.get('created_at')),
                '{tvm_id}{tvm_app}'.format(
                    tvm_id=row.get('tvm_client_id', ''),
                    tvm_app=' ({})'.format(row['tvm_app'].get('name')) if 'tvm_app' in row else '',
                ),
                row.get('signature', ''),
                row.get('comment', ''),
                row.get('token_uuid', ''),
                row.get('state_name', '')
            ) for row in tokens
        ],
        header=['created', 'tvm_client_id', 'signature', 'comment', 'token_uuid', 'state'],
        compact=compact,
    )


def _escape_java_string(str_, type_='value'):
    if type_ == 'property':
        return ESCAPE_JAVA_PROP_RE.sub(r'\\\1', str_)
    else:
        return ESCAPE_JAVA_VALUE_RE.sub(r'\\\1', str_)


def value_to_java_properties(dict_):
    # https://en.wikipedia.org/wiki/.properties (VAULT-165)
    return '\n'.join(
        [
            u'{key} = {value}'.format(
                key=_escape_java_string(k, 'property'),
                value=_escape_java_string(v, 'value'),
            )
            for k, v in dict_.items()
        ],
    ).encode('latin1', 'backslashreplace')


def value_to_yaml(dict_):
    return smart_unicode(
        yaml.safe_dump(
            dict_,
            default_flow_style=False,
            indent=4,
            allow_unicode=True,
        )
    )
