# coding: utf-8

from .base import VaultCLICommand
from .format import (
    format_table,
    format_timestamp,
)


class ListSecretsCommand(VaultCLICommand):
    """
    List of secrets (without values)
    """
    def __init__(self, args_parser, *args, **kwargs):
        super(ListSecretsCommand, self).__init__(args_parser, *args, **kwargs)
        self.usage = '%(prog)s [-t <tags>] [-p page]'
        self.add_base_argument('query')
        self.add_base_argument('query_type')
        self.add_base_argument('tags')
        self.add_base_argument('page')

    def process(self, cli_args, client, debug=False, *args, **kwargs):
        response = client.list_secrets(
            query=cli_args.query,
            query_type=cli_args.query_type,
            tags=cli_args.tags,
            page=cli_args.page,
        )
        self.print_response_and_exit(response)

    def serialize_success_response(self, response):
        result = [
            (
                v['uuid'],
                v['name'],
                v.get('last_secret_version', {}).get('version', ''),
                v.get('versions_count', '—'),
                v.get('tokens_count', '—'),
                v.get('effective_role', ''),
                format_timestamp(v['created_at']),
            )
            for v in response.result
        ]
        return format_table(
            result,
            header=['secret uuid', 'name', 'last version uuid', 'cnt', 'tok', 'acl', 'created'],
            compact=self.compact,
        )


class CreateSecretCommand(VaultCLICommand):
    """
    Create a new secret

    Create new secret authorizing user by SSH private keys or OAuth tokens.
    SSH validation conducted through signature validation by public keys retrieved
    from https://staff.yandex-team.ru/
    """
    def __init__(self, args_parser, *args, **kwargs):
        super(CreateSecretCommand, self).__init__(args_parser, *args, **kwargs)
        self.usage = '%(prog)s name [-v <json>] [-r <role> ...] [-c <comment>] [-t <tags>]'
        self.add_argument('name', help='Non unique secret name')
        self.add_base_argument('value')
        self.add_base_argument('kv')
        self.add_base_argument('files_kv')
        self.add_base_argument('roles')
        self.add_base_argument('secret_comment')
        self.add_base_argument('tags')
        self.add_base_argument('version_ttl')

    def process(self, cli_args, client, debug=False, *args, **kwargs):
        response = client.create_secret(
            cli_args.name,
            value=self.get_value(cli_args, skip_exception=True),
            comment=cli_args.comment,
            tags=cli_args.tags,
            roles=cli_args.roles,
            version_ttl=cli_args.version_ttl,
        )
        self.print_response_and_exit(response)

    def serialize_success_response(self, response):
        return self.serialize_secret_response(response)


class UpdateSecretCommand(VaultCLICommand):
    """
    Update secret metainformation and versions
    """
    def __init__(self, args_parser, *args, **kwargs):
        super(UpdateSecretCommand, self).__init__(args_parser, *args, **kwargs)
        self.usage = '%(prog)s secret_uuid [-v <json>] [-r <role> ...] [-c <comment>] [-t <tags>]'
        self.add_base_argument('secret_uuid')
        self.add_base_argument('secret_name')
        self.add_base_argument('secret_state')
        self.add_base_argument('value')
        self.add_base_argument('kv')
        self.add_base_argument('files_kv')
        self.add_base_argument('update_value')
        self.add_base_argument('delete_keys')
        self.add_base_argument('roles')
        self.add_base_argument('secret_comment')
        self.add_base_argument('tags')
        self.add_base_argument('version_ttl')

    def serialize_success_response(self, response):
        return self.serialize_secret_response(response)

    def process(self, cli_args, client, debug=False, *args, **kwargs):
        response = client.update_secret(
            cli_args.secret_uuid,
            name=cli_args.name,
            value=self.get_value(cli_args, skip_exception=True),
            update=cli_args.update,
            comment=cli_args.comment,
            tags=cli_args.tags,
            roles=cli_args.roles,
            state=cli_args.secret_state,
            version_ttl=cli_args.version_ttl,
        )
        self.print_response_and_exit(response)


class GetSecretCommand(VaultCLICommand):
    """
    Get secret metainformation
    """
    def __init__(self, args_parser, *args, **kwargs):
        super(GetSecretCommand, self).__init__(args_parser, *args, **kwargs)
        self.usage = '%(prog)s secret_uuid'
        self.add_base_argument('secret_uuid')

    def process(self, cli_args, client, debug=False, *args, **kwargs):
        response = client.get_secret(cli_args.secret_uuid)
        self.print_response_and_exit(response)

    def serialize_success_response(self, response):
        return self.serialize_secret_response(response)
