# coding: utf-8

import textwrap

from .base import VaultCLICommand
from .format import (
    format_comment,
    format_creator,
    format_role,
    format_timestamp,
    format_tokens,
)


class BaseTokenCommand(VaultCLICommand):
    def format_tvm_app(self, token_info):
        result = str(token_info.get('tvm_client_id', '-'))
        tvm_app = token_info.get('tvm_app')
        if tvm_app:
            result = u'{tvm_client_id} ({name}, {state}, {abc_department})'.format(
                tvm_client_id=tvm_app['tvm_client_id'],
                name=tvm_app.get('name', ''),
                state=tvm_app.get('abc_state', ''),
                abc_department=tvm_app.get('abc_department', {}).get('display_name', ''),
            )
        return result

    def serialize_token_response(self, response):
        token_info = response.result.get('token_info', {})
        return textwrap.dedent(u'''
            secret_uuid: {secret_uuid}
            token uuid: {token_uuid}
            state: {state_name}{revocation}
            comment: {comment}

            tvm app: {tvm_app}

            creator: {creator}
            created: {created}

            owners:
            {owners}

            readers:
            {readers}
        ''').format(
            secret_uuid=token_info.get('secret_uuid') or u'-',
            token_uuid=token_info['token_uuid'],
            state_name=token_info['state_name'],
            created=format_timestamp(token_info.get('created_at')),
            creator=format_creator(token_info),
            comment=format_comment(token_info.get('comment', '')),
            tvm_app=self.format_tvm_app(token_info),
            owners=u'\n'.join(
                (format_role(row) for row in response.result['owners']),
            ) or u'-',
            readers=u'\n'.join(
                (format_role(row) for row in response.result.get('readers', [])),
            ) or u'-',
            revocation=' ({revocation_login}, {revoked_timestamp})'.format(
                revocation_login=token_info.get('revocation_login', u'-'),
                revoked_timestamp=format_timestamp(token_info.get('revoked_at')),
            ) if token_info['state_name'] == 'revoked' else '',
        ).lstrip()

    def serialize_success_response(self, response):
        return self.serialize_token_response(response)


class ListTokensCommand(BaseTokenCommand):
    """
    List of tokens information
    """
    def __init__(self, args_parser, *args, **kwargs):
        super(ListTokensCommand, self).__init__(args_parser, *args, **kwargs)
        self.usage = '%(prog)s secret_uuid'
        self.add_base_argument('secret_uuid')
        self.add_base_argument('with_revoked')
        self.add_base_argument('page')

    def process(self, cli_args, client, debug=False, *args, **kwargs):
        response = client.list_tokens(
            cli_args.secret_uuid,
            with_revoked=cli_args.with_revoked,
            page=cli_args.page,
        )
        self.print_response_and_exit(response)

    def serialize_success_response(self, response):
        return format_tokens(response.result, compact=self.compact)


class CreateTokenCommand(BaseTokenCommand):
    """
    Create a new token for a secret
    """
    def __init__(self, args_parser, *args, **kwargs):
        super(CreateTokenCommand, self).__init__(args_parser, *args, **kwargs)
        self.usage = '%(prog)s secret_uuid [-s <signature>] [-tvm <tvm-client-id>]'
        self.add_base_argument('secret_uuid')
        self.add_base_argument('token_signature')
        self.add_base_argument('token_tvm_client_id')
        self.add_base_argument('token_comment')

    def process(self, cli_args, client, debug=False, *args, **kwargs):
        response = client.create_token(
            cli_args.secret_uuid,
            tvm_client_id=cli_args.tvm_client_id,
            signature=cli_args.signature,
            comment=cli_args.comment,
        )
        self.print_response_and_exit(response)

    def serialize_success_response(self, response):
        return textwrap.dedent('''
            secret uuid: {secret_uuid}
            token: {token}
            token uuid: {token_uuid}
        ''').lstrip().format(
            token=response.result.get('token', ''),
            token_uuid=response.result.get('token_uuid', ''),
            secret_uuid=response.result.get('secret_uuid', ''),
        )


class FindTokenCommand(BaseTokenCommand):
    """
    Get token info by token or token_uuid
    """
    def __init__(self, args_parser, *args, **kwargs):
        super(FindTokenCommand, self).__init__(args_parser, *args, **kwargs)
        self.usage = '%(prog)s token_data'
        self.add_argument('token_data', help='Token or token UUID')

    def process(self, cli_args, client, debug=False, *args, **kwargs):
        token_query = dict()
        if cli_args.token_data[:4].lower() == 'tid-':
            token_query['token_uuid'] = cli_args.token_data
        else:
            token_query['token'] = cli_args.token_data

        response = client.get_token_info(**token_query)
        self.print_response_and_exit(response)


class RevokeTokenCommand(BaseTokenCommand):
    """
    Revoke token by token_uuid
    """
    def __init__(self, args_parser, *args, **kwargs):
        super(RevokeTokenCommand, self).__init__(args_parser, *args, **kwargs)
        self.usage = '%(prog)s token_uuid'
        self.add_base_argument('token_uuid')

    def process(self, cli_args, client, debug=False, *args, **kwargs):
        response = client.revoke_token(cli_args.token_uuid)
        self.print_response_and_exit(response)


class RestoreTokenCommand(BaseTokenCommand):
    """
    Restore revoked token by token_uuid
    """
    def __init__(self, args_parser, *args, **kwargs):
        super(RestoreTokenCommand, self).__init__(args_parser, *args, **kwargs)
        self.usage = '%(prog)s token_uuid'
        self.add_base_argument('token_uuid')

    def process(self, cli_args, client, debug=False, *args, **kwargs):
        response = client.restore_token(cli_args.token_uuid)
        self.print_response_and_exit(response)
