# coding: utf-8

from base64 import b64decode
import json

import six

from .base import (
    JSON_DUMPS_DEFAULT_ARGS,
    VaultCLICommand,
)
from .format import (
    value_to_java_properties,
    value_to_yaml,
)


class BaseVersionCommand(VaultCLICommand):
    def __init__(self, args_parser, *args, **kwargs):
        super(BaseVersionCommand, self).__init__(args_parser, *args, **kwargs)

        self.add_base_argument('only_value')
        self.add_base_argument('skip_nl')
        self.add_base_argument('value_format')
        self.only_value = False
        self.only_value_key = None
        self.skip_nl = False

    def process_args(self, cli_args):
        cli_args = super(BaseVersionCommand, self).process_args(cli_args)
        self.only_value_key = cli_args.only_value
        self.only_value = self.only_value_key != ''
        self.skip_nl = cli_args.skip_nl
        if cli_args.value_format is not None:
            self.value_format = cli_args.value_format
        return cli_args

    def _decode_value(self, value, encoding='base64'):
        if encoding == 'base64':
            return b64decode(value)
        return value

    def serialize_success_response(self, response):
        if self.only_value:
            value = response.result.get('value')
            if isinstance(self.only_value_key, six.string_types) and len(self.only_value_key) > 0:
                for v in value:
                    if v['key'] == self.only_value_key:
                        encoding = v.get('encoding')
                        decoded_value = self._decode_value(v['value'], encoding)
                        if encoding:
                            return decoded_value
                        else:
                            return decoded_value + ('' if self.skip_nl else '\n')
                raise KeyError('The key \'{}\' is not found in the value.'.format(self.only_value_key))
            else:
                packed_value = self._pack_value(value)
                if self.value_format == 'java':
                    return value_to_java_properties(packed_value) + '\n'
                if self.value_format == 'yaml':
                    return value_to_yaml(packed_value)
                return json.dumps(packed_value, **JSON_DUMPS_DEFAULT_ARGS) + '\n'
        else:
            return self.serialize_version_response(response)

    def serialize_success_response_as_json(self, response):
        value = response.result.get('value')
        if self.only_value:
            value = self._pack_value(value)
        if self.only_value and isinstance(self.only_value_key, six.string_types) and len(self.only_value_key) > 0:
            value = {self.only_value_key: value[self.only_value_key]}

        return json.dumps(
            value if self.only_value else response.result,
            **JSON_DUMPS_DEFAULT_ARGS
        ) + '\n'


class CreateVersionCommand(BaseVersionCommand):
    """
    Create a new secret version
    """
    def __init__(self, args_parser, *args, **kwargs):
        super(CreateVersionCommand, self).__init__(args_parser, *args, **kwargs)
        self.usage = '%(prog)s secret_uuid <json value> [-u] [-r <role> ...]'
        self.add_base_argument('secret_uuid')
        self.add_base_argument('value')
        self.add_base_argument('kv')
        self.add_base_argument('files_kv')
        self.add_base_argument('update_value')
        self.add_base_argument('delete_keys')
        self.add_base_argument('roles')
        self.add_base_argument('version_comment')
        self.add_base_argument('version_ttl')

    def process(self, cli_args, client, debug=False, *args, **kwargs):
        response = client.create_version(
            cli_args.secret_uuid,
            value=self.get_value(cli_args),
            update=cli_args.update,
            comment=cli_args.comment,
            roles=cli_args.roles,
            packed_value=not self.only_value,
            ttl=cli_args.version_ttl,
        )
        self.print_response_and_exit(response)

    def print_response_and_exit(self, response):
        if response.success and isinstance(response.result, six.string_types):
            self.echo(u'New version uuid is {}'.format(response.result), err=True)
        else:
            super(CreateVersionCommand, self).print_response_and_exit(response)


class UpdateVersionCommand(BaseVersionCommand):
    """
    Update secret version metadata
    """
    def __init__(self, args_parser, *args, **kwargs):
        super(UpdateVersionCommand, self).__init__(args_parser, *args, **kwargs)
        self.usage = '%(prog)s version_uuid [--ttl 30] [--sate hidden|normal]'
        self.add_base_argument('version_uuid')
        self.add_base_argument('version_ttl')
        self.add_base_argument('version_state')
        self.add_base_argument('version_comment')

    def process(self, cli_args, client, debug=False, *args, **kwargs):
        response = client.update_version(
            cli_args.version_uuid,
            state=cli_args.version_state,
            ttl=cli_args.version_ttl,
            comment=cli_args.comment,
        )
        self.print_response_and_exit(response)


class GetVersionCommand(BaseVersionCommand):
    """
    Get a secret value
    """
    def __init__(self, args_parser, *args, **kwargs):
        super(GetVersionCommand, self).__init__(args_parser, *args, **kwargs)
        self.usage = '%(prog)s <uuid>'
        self.add_base_argument('uuid')

    def process(self, cli_args, client, debug=False, *args, **kwargs):
        response = client.get_version(
            cli_args.uuid,
            packed_value=not self.only_value,
        )
        self.print_response_and_exit(response)
