# coding: utf-8

import argparse
import os
import re
import sys
import warnings

import argcomplete
from vault_client_cli.cli_base import (
    CLIManager,
    Namespace,
)
from vault_client_cli.commands import (
    CreateSecretCommand,
    CreateTokenCommand,
    CreateVersionCommand,
    FindTokenCommand,
    GetSecretCommand,
    GetVersionCommand,
    ListSecretsCommand,
    ListTokensCommand,
    OAuthCommand,
    RestoreTokenCommand,
    RevokeTokenCommand,
    TLSCreateCommand,
    TLSRotateCommand,
    UpdateSecretCommand,
    UpdateVersionCommand,
)


YAV_BACKEND_VAR = 'YAV_BACKEND'
YAV_BACKEND_DEBUG_VAR = 'YAV_DEBUG'
YAV_OAUTH_TOKEN_VAR = 'YAV_TOKEN'

ENV_TRUE_VALUES = ('1', 'true', 'yes')


class VersionAction(argparse.Action):
    def __init__(self, option_strings, program=None, version=None,
                 dest=argparse.SUPPRESS, default=argparse.SUPPRESS,
                 help="show program's version number and exit"
                 ):
        super(VersionAction, self).__init__(
            option_strings=option_strings,
            dest=dest,
            default=default,
            nargs=0,
            help=help,
        )
        self.program = program or os.path.basename(sys.argv[0])

    def __call__(self, parser, namespace, values, option_string=None):
        try:
            from library.python.vault_client import __version__ as version
        except ImportError:
            from vault_client import __version__ as version

        parser.exit(message='{} {}\n'.format(
            self.program,
            '{yav} (python: {py})'.format(
                yav=version,
                py=sys.version.replace('\n', ''),
            )
        ))


class VaultCLIManager(CLIManager):
    def __init__(self, client=None, doc=None, stdout=None, stderr=None):
        super(VaultCLIManager, self).__init__(
            doc=doc,
            stdout=stdout,
            stderr=stderr,
            debug=(os.environ.get(YAV_BACKEND_DEBUG_VAR, '').lower() in ENV_TRUE_VALUES),
        )
        self.client = client
        self.commands_kwargs = dict()

        self.commands_kwargs['debug'] = self.debug

        self.add_argument('--version', program='yav', action=VersionAction)

        self.assign_command('list secrets', ListSecretsCommand)
        self.assign_command('create secret', CreateSecretCommand)
        self.assign_command('get secret', GetSecretCommand)
        self.assign_command('update secret', UpdateSecretCommand)

        self.assign_command('create version', CreateVersionCommand)
        self.assign_command('get version', GetVersionCommand)
        self.assign_command('update version', UpdateVersionCommand)

        self.assign_command('list tokens', ListTokensCommand)
        self.assign_command('create token', CreateTokenCommand)
        self.assign_command('find token', FindTokenCommand)
        self.assign_command('revoke token', RevokeTokenCommand)
        self.assign_command('restore token', RestoreTokenCommand)

        self.assign_command('oauth', OAuthCommand)

        self.assign_command('tls create', TLSCreateCommand)
        self.assign_command('tls rotate', TLSRotateCommand)

    def parse_args(self, args, **kwargs):
        if args is None:
            args = sys.argv[1:]  # pragma: no cover

        argcomplete.autocomplete(self.args_parser)

        with self.process_errors(exit_code=2, print_help=True, args=args):
            # Хак. Если параметр начинается с минуса и содержит команду для роли,
            # то добавляем в начале пробел, чтобы argparse не валился на разборе
            role_re = re.compile('^[-+][^:]+:')
            args = [
                ' ' + v if role_re.search(v) else v
                for v in args
            ]
            args = super(VaultCLIManager, self).parse_args(args, **kwargs)

        return args

    def build_client(self, args, native_client=None):
        from vault_client_cli.client import (
            VaultClient,
            CLIVaultClient,
            VAULT_PRODUCTION_API,
            VAULT_TESTING_API,
        )
        vault_client_args = dict(
            native_client=native_client,
            decode_files=False,
        )

        host = os.environ.get(YAV_BACKEND_VAR, '').strip().rstrip('/')
        oauth_token = args.oauth_token or os.environ.get(YAV_OAUTH_TOKEN_VAR)

        if oauth_token:
            vault_client_args['rsa_auth'] = False
            vault_client_args['authorization'] = 'OAuth {}'.format(oauth_token)
        else:
            rsa_login, rsa_auth = self._get_rsa_auth_from_args(args)
            vault_client_args['rsa_auth'] = rsa_auth
            vault_client_args['rsa_login'] = rsa_login

        if host:
            vault_client = VaultClient(host=host, check_status=False, **vault_client_args)
        elif args.testing:
            vault_client = VaultClient(host=VAULT_TESTING_API, **vault_client_args)
        else:
            vault_client = VaultClient(host=VAULT_PRODUCTION_API, **vault_client_args)

        return CLIVaultClient(vault_client=vault_client)

    def process(self, args=None, commands_kwargs=None):
        parsed_args = self.parse_args(args, namespace=Namespace())
        with warnings.catch_warnings():
            if parsed_args.skip_warnings:
                warnings.simplefilter('ignore')

            if not self.client:  # pragma: no cover
                self.client = self.build_client(parsed_args)
            self.commands_kwargs['client'] = self.client

            super(VaultCLIManager, self).process(
                args=args,
                commands_kwargs=self.commands_kwargs,
            )
