# coding: utf-8

import base64
from collections import OrderedDict
import fnmatch
import os
import re

import jinja2
import six


try:
    from library.python.vault_client.utils import merge_dicts
except ImportError:
    from vault_client.utils import merge_dicts


class ActionError(Exception):
    def __init__(self, action, message=''):
        super(ActionError, self).__init__(message)
        self.action = action

    def __str__(self):
        return u'{action}: {message}'.format(
            action=repr(self.action),
            message=super(Exception, self).__str__(),
        )


class BaseAction(object):
    source_re = None
    target_re = None

    def __init__(self, section, secrets, target, source):
        self.section = section
        self.environment = self.section.environment
        self.secrets = secrets
        self.target = target
        self.source = source

        self.need_post_update = False

        self.parse_target()
        self.parse_source()

    def __repr__(self):
        return '{self.__class__.__name__}<{self.target} = {self.source}>'.format(self=self)

    def __str__(self):
        return '[{section.name}] {self.target} = {self.source}'.format(section=self.section, self=self)

    def parse_target(self):
        pass

    def parse_source(self):
        pass

    def prepare(self):
        pass

    def commit(self, force=False):
        pass

    @classmethod
    def build_action(cls, environment, secrets, target, source, *args, **kwargs):
        if cls.source_re and not cls.source_re.match(source):
            return
        if cls.target_re and not cls.target_re.match(target):
            return
        return cls(environment, secrets, target, source, *args, **kwargs)


class BaseFileAction(BaseAction):
    source_re = re.compile(r'^(?P<uuid>(?:sec|ver)-[0-9a-z]{26,})(?:\[(?P<keys>.+?)\])?$', re.I)

    def fetch_secret(self):
        self.data = self.pack_secret_value(
            self.secrets.get(self.secret_uuid, packed_value=False),
        )

        if self.keys:
            result = set()
            keys = self.data.keys()
            for k in self.keys:
                found_keys = fnmatch.filter(keys, k)
                if not found_keys:
                    raise ActionError(
                        self,
                        'The keys matched the "{key}" template is not found in the secret "{secret_uuid}"'.format(
                            key=k,
                            secret_uuid=self.secret_uuid,
                        ),
                    )
                result.update(found_keys)
            self.data = dict(map(lambda x: (x, self.data[x]), result))

    def pack_secret_value(self, value):
        result = OrderedDict()
        for v in value:
            processed_value = v['value']
            encoding = v.get('encoding')
            if encoding and encoding == 'base64':
                processed_value = base64.b64decode(processed_value)
            result[v['key']] = processed_value
        return result

    def parse_source(self):
        matches = self.source_re.match(self.source.strip())
        if not matches:
            raise ActionError(self, '{} is an invalid source value'.format(self.source))
        self.secret_uuid = matches.group('uuid')
        self.keys = [s.strip() for s in matches.group('keys').split(',')] if matches.group('keys') else None

        self.fetch_secret()


class UnknownFileAction(BaseAction):
    target_re = re.compile(r'^/')

    def parse_source(self):
        raise ActionError(self, 'Unknown file action: {} = {}'.format(self.target, self.source))


class SingleFileAction(BaseFileAction):
    target_re = re.compile(r'^(?P<filename>/[^\s\*:]+?)(?:\:(?P<mode>\d+))?$')

    def parse_source(self):
        super(SingleFileAction, self).parse_source()
        if not self.keys or len(self.keys) != 1:
            raise ActionError(self, 'The source must contain a single key value')

    def parse_target(self):
        matches = self.target_re.match(self.target)
        self.filename = matches.group('filename')
        self.mode = matches.group('mode')
        if self.mode is not None:
            self.mode = int(self.mode, 8)

    def prepare(self):
        self.prepared_data = self.data.get(self.keys[0])
        if not self.prepared_data:
            raise ActionError(self, 'File not found in a secret')

    def commit(self, force=False):
        filename = self.section.path_join(self.filename)
        self.need_post_update = self.environment.save_file(
            filename,
            self.prepared_data,
            force=force,
            permissions=self.mode if self.mode is not None else self.section.mode,
            owner=self.section.owner,
        )


class MultiFileAction(BaseFileAction):
    target_re = re.compile(r'^(?P<path>/\S*?)[\.*\$](?:\:(?P<mode>\d+))?$')
    source_re = re.compile(
        r'(?:\s*(?:sec|ver)-[0-9a-z]{26,}(?:\s*\[.+?\])?[\s,]?)+$',
        re.I,
    )
    secret_re = re.compile(
        r'(?P<uuid>(?:sec|ver)-[0-9a-z]{26,})(?:\s*\[(?P<keys>.+?)\])?',
        re.I,
    )

    def parse_source(self):
        self.action_secrets = self.parse_secrets(self.source)
        self.fetch_secrets()

    def parse_secrets(self, action_secrets):
        result = OrderedDict()
        for secret in self.secret_re.finditer(action_secrets):
            secret_uuid = secret.group('uuid')
            if secret_uuid in result:
                self.environment.logger.debug(
                    '[{section.name}] Duplicate a secret uuid ({secret_uuid}) '
                    'in an action source ({action_secrets})'.format(
                        section=self.section,
                        secret_uuid=secret_uuid,
                        action_secrets=action_secrets,
                    )
                )

            secret_data = result.setdefault(
                secret_uuid,
                dict(
                    keys=set(),
                ),
            )
            secret_data['keys'].update(
                map(
                    lambda x: x.strip(),
                    secret.group('keys').split(',') if secret.group('keys') else [],
                ),
            )
        return result

    def fetch_secrets(self):
        self.data = OrderedDict()
        for secret_uuid, meta in self.action_secrets.items():
            secret_data = self.pack_secret_value(
                self.secrets.get(secret_uuid, packed_value=False),
            )
            if meta['keys']:
                result = set()
                keys = secret_data.keys()
                for k in meta['keys']:
                    found_keys = fnmatch.filter(keys, k)
                    if not found_keys:
                        raise ActionError(
                            self,
                            'The keys matched the "{key}" template is not found in the secret "{secret_uuid}"'.format(
                                key=k,
                                secret_uuid=secret_uuid,
                            ),
                        )
                    result.update(found_keys)
                secret_data = dict(map(lambda x: (x, secret_data[x]), result))

            self.data.update(secret_data)

    def parse_target(self):
        matches = self.target_re.match(self.target)
        self.path = matches.group('path')
        self.mode = matches.group('mode')
        if self.mode is not None:
            self.mode = int(self.mode, 8)

    def prepare(self):
        self.prepared_data = [
            (self.section.path_join(self.path, os.path.basename(k)), self.data[k])
            for k in self.data.keys()
        ]
        self.prepared_data.sort(key=lambda x: x[0])
        if not self.prepared_data:
            raise ActionError(self, 'Files not found in a secret')

    def commit(self, force=False):
        for f in self.prepared_data:
            result = self.environment.save_file(
                f[0],
                f[1],
                force=force,
                permissions=self.mode if self.mode is not None else self.section.mode,
                owner=self.section.owner,
            )
            self.need_post_update = self.need_post_update or result


class FileTemplateAction(BaseFileAction):
    target_re = re.compile(r'^(?P<filename>/[^\s*:]+?)(?:\:(?P<mode>\d+))?$')
    source_re = re.compile(
        r'^.+?:(?:\s*(?:sec|ver)-[0-9a-z]{26,}(?:\s*->\s*\w+)?(?:\s*\[(?P<keys>.+?)\])?[\s,]?)+$',
        re.I,
    )
    secret_re = re.compile(
        r'(?P<uuid>(?:sec|ver)-[0-9a-z]{26,})(?:\s*->\s*(?P<alias>\w+))?(?:\s*\[(?P<keys>.+?)\])?',
        re.I,
    )

    jenv_prefix = 'jenv:'

    def parse_target(self):
        matches = self.target_re.match(self.target)
        self.filename = matches.group('filename')
        self.mode = matches.group('mode')
        if self.mode is not None:
            self.mode = int(self.mode, 8)

    def parse_source(self):
        self.template, self.template_secrets = self.source.strip().split(':', 1)
        self.template_secrets = self.parse_secrets(self.template_secrets)
        self.fetch_secrets()

    def parse_secrets(self, template_secrets):
        result = OrderedDict()
        for secret in self.secret_re.finditer(template_secrets):
            secret_uuid = secret.group('uuid')
            if secret_uuid in result:
                self.environment.logger.debug(
                    '[{section.name}] Duplicate a secret uuid ({secret_uuid}) '
                    'in an action source ({template_secrets})'.format(
                        section=self.section,
                        secret_uuid=secret_uuid,
                        template_secrets=template_secrets,
                    )
                )

            secret_data = result.setdefault(
                secret_uuid,
                dict(
                    aliases=set(),
                    keys=set(),
                    add_keys=False,
                ),
            )
            if secret.group('alias'):
                secret_data['aliases'].add(secret.group('alias'))
            else:
                secret_data['add_keys'] = True

            secret_data['keys'].update(
                map(
                    lambda x: x.strip(),
                    secret.group('keys').split(',') if secret.group('keys') else [],
                ),
            )
        return result

    def fetch_secrets(self):
        self.data = OrderedDict()
        for secret_uuid, meta in self.template_secrets.items():
            unpacked_secret_data = self.secrets.get(secret_uuid, packed_value=False)
            secret_data = self.pack_secret_value(unpacked_secret_data)

            for row in unpacked_secret_data:
                if row.get('encoding', '').lower() == 'base64':
                    self.environment.logger.debug(
                        '[{section.name}] Warning: {secret_uuid}[{key}] contains a binary file. Don\'t use this key in the template'.format(
                            section=self.section,
                            secret_uuid=secret_uuid,
                            key=row['key'],
                        ),
                    )

            if meta['keys']:
                result = set()
                keys = secret_data.keys()
                for k in meta['keys']:
                    found_keys = fnmatch.filter(keys, k)
                    if not found_keys:
                        raise ActionError(
                            self,
                            'The keys matched the "{key}" template is not found in the secret "{secret_uuid}"'.format(
                                key=k,
                                secret_uuid=secret_uuid,
                            ),
                        )
                    result.update(found_keys)
                secret_data = dict(map(lambda x: (x, secret_data[x]), result))

            if meta['aliases']:
                for k in meta['aliases']:
                    self.data[k] = secret_data

            if meta['add_keys']:
                self.data.update(secret_data)

    def _get_default_template_vars(self):
        return dict(
            YENV_TYPE=self.environment.type,
            YENV_NAME=self.environment.name,
            YAV_DEPLOY_HOSTNAME=self.environment.hostname,
            YAV_DEPLOY_TAGS=self.environment.tags,
        )

    def get_jenv(self):
        prefix_len = len(self.jenv_prefix)
        return dict([
            (k[prefix_len:], v) for k, v in six.iteritems(self.section.vars) if k.startswith(self.jenv_prefix)
        ])

    def process_template(self):
        return self.environment.templates.get_template(self.template).render(merge_dicts(
            self._get_default_template_vars(),
            self.get_jenv(),
            self.data,
        ))

    def prepare(self):
        try:
            self.prepared_data = self.process_template().encode('utf-8')
        except jinja2.TemplateError as e:
            raise ActionError(self, u'{e.__class__.__name__} {e}'.format(e=e))

    def commit(self, force=False):
        filename = self.section.path_join(self.filename)
        self.need_post_update = self.environment.save_file(
            filename,
            self.prepared_data,
            force=force,
            permissions=self.mode if self.mode is not None else self.section.mode,
            owner=self.section.owner,
        )
