# coding: utf-8

import argparse
import logging
import os
import platform
import re
import sys

import argcomplete
import six
from vault_client_deploy.configs import Environment


YAV_DEPLOY_CONFS_PATH = '/etc/yandex/yav-deploy'
YAV_DEPLOY_PKGS_PREFIX = 'pkg/'
YAV_DEPLOY_ROOT = '/'

OK_STATUS = 0
HAS_ERRORS_STATUS = 3
BAD_ARGUMENTS_STATUS = 2


def get_version():
    try:
        from library.python.vault_client import __version__ as version
    except ImportError:
        from vault_client import __version__ as version
    return version


class VersionAction(argparse.Action):
    def __init__(self, option_strings, program=None, version=None,
                 dest=argparse.SUPPRESS, default=argparse.SUPPRESS,
                 help="show program's version number and exit"
                 ):
        super(VersionAction, self).__init__(
            option_strings=option_strings,
            dest=dest,
            default=default,
            nargs=0,
            help=help,
        )
        self.program = program or os.path.basename(sys.argv[0])

    def __call__(self, parser, namespace, values, option_string=None):
        parser.exit(message='{} {}\n'.format(
            self.program,
            '{yav_deploy} (python: {py})'.format(
                yav_deploy=get_version(),
                py=sys.version.replace('\n', ''),
            ),
        ))


class VaultDeployArgsParser(argparse.ArgumentParser):
    def __init__(self, version=None, *args, **kwargs):
        super(VaultDeployArgsParser, self).__init__(
            formatter_class=argparse.RawDescriptionHelpFormatter,
            *args,
            **kwargs
        )

        self.epilog = '\n\nRead the documentation on the website https://vault-api.passport.yandex.net/docs/'

        self.add_argument('--version', '-V', program='yav-deploy', action=VersionAction)

        self.add_argument(
            '-f', '--force', dest='force', action='store_true', help='Force overwrite unmodified files',
        )
        self.add_argument(
            '--rsa-login', dest='rsa_login', metavar='login', help='RSA private key login.', default=None,
        )
        self.add_argument(
            '--rsa-private-key',
            dest='rsa_private_key', metavar='FILENAME', help='RSA private key filename.', default=None,
        )
        self.add_argument(
            '--rsa-agent-key-hash', dest='rsa_agent_key_hash', metavar='KEY_HASH', default=None,
            help='''
                SSH key hash from the ssh agent. You can get the list of keys in the agent by the command 'ssh-add -l'
            ''',
        )
        self.add_argument(
            '--oauth', dest='oauth_token', metavar='token', default=None,
            help='OAuth token. Alternative: put the token in the YAV_TOKEN environment variable. '
                 'Get token via "yav oauth" command or at https://nda.ya.ru/3UVsAN',
        )
        self.add_argument(
            '-c', '--configs-path', dest='confs_path', default=YAV_DEPLOY_CONFS_PATH,
            metavar='path',
            help='Environment configs root folder (default: {})'.format(YAV_DEPLOY_CONFS_PATH),
        )
        self.add_argument(
            '--skip-pkg', dest='skip_pkg', action='store_true',
            help='Do not find configs in the pkg subfolder',
        )
        self.add_argument(
            '-d', '--deploy-root', dest='deploy_root', default=YAV_DEPLOY_ROOT,
            metavar='path',
            help='Deploy root folder (default: {})'.format(YAV_DEPLOY_ROOT),
        )
        self.add_argument(
            '--file', dest='config_files', nargs='*', metavar='filename',
            help='Only process the specified configuration files',
        )
        self.add_argument(
            '--sections', dest='sections', nargs='*', metavar='section name',
            help='Only process the specified section in the configuration files',
        )
        self.add_argument(
            '--skip-chown', dest='skip_chown', action='store_true',
            help='Skip chown for temp files',
        )
        self.add_argument(
            '--skip-pre-update', dest='skip_pre_update', action='store_true',
            help='Skip pre update commands',
        )
        self.add_argument(
            '--skip-post-update', dest='skip_post_update', action='store_true',
            help='Skip post update commands',
        )
        self.add_argument(
            '--quiet', dest='quiet', action='store_true', help='Suppress all error messages',
        )
        self.add_argument(
            '--testing', dest='testing', action='store_true', help='Send requests to a testing environment',
        )
        self.add_argument(
            '--debug', dest='debug', action='store_true', help='Print debug info',
        )

    def error(self, message, stderr=None):
        stderr = stderr or sys.stderr
        six.print_('error: %s' % message, file=stderr)
        six.print_('-' * 20, file=stderr)
        self.print_help(file=stderr)
        sys.exit(2)


class VaultDeployApp(object):
    def __init__(self, doc=None):
        self.args_parser = VaultDeployArgsParser(description=doc)
        self.confs_path = None
        self.parsed_args = None
        self.environments = None

    def add_argument(self, *args, **kwargs):
        self.args_parser.add_argument(*args, **kwargs)

    def parse_args(self, argv=None):
        result = self.args_parser.parse_args(argv)
        return result

    def exit(self, code=OK_STATUS):
        sys.exit(code)

    def log_version(self, only_debug=True):
        is_arcadia_python = getattr(sys, "is_standalone_binary", False)

        build = 'External Python'
        if is_arcadia_python:
            rev = 'unknown'
            try:
                import library.python.svn_version as sv
                rev = sv.svn_revision()
            except ImportError:
                pass
            build = 'Arcadia Python (svn rev.: {})'.format(rev)

        version = '{program} {version} ({build} {system}; {platform}; {node}) {p_impl}/{p_ver} ({p_comp})'.format(
            program=os.path.basename(sys.argv[0]),
            version=get_version(),
            build=build,
            system=platform.system(),
            platform=platform.platform(),
            node=platform.node(),
            p_impl=platform.python_implementation(),
            p_ver=platform.python_version(),
            p_comp=platform.python_compiler(),
        )
        self.logger.debug(version)

    def log_argv(self, argv):
        if argv is None:
            return
        s = re.sub(
            r'(--oauth[=\s]+)[\S]+\b',
            r'\1*****',
            ' '.join(argv),
            flags=re.IGNORECASE,
        )
        self.logger.debug(s)

    def find_environments(self, confs_path, parsed_args, skip_pkg=False):
        result = [
            self.create_environment(confs_path, parsed_args),
        ]

        pkg_confs_path = os.path.join(confs_path, YAV_DEPLOY_PKGS_PREFIX)
        if not skip_pkg:
            self.logger.debug('Find packages environements in {}'.format(pkg_confs_path))
            found_pkg_environments = 0
            if os.path.exists(pkg_confs_path):
                dirs = sorted(filter(
                    lambda x: os.path.isdir(x),
                    map(
                        lambda x: os.path.join(pkg_confs_path, x),
                        os.listdir(pkg_confs_path),
                    ),
                ))
                for d in dirs:
                    result.append(
                        self.create_environment(d, parsed_args),
                    )
                    found_pkg_environments += 1
            if not found_pkg_environments:
                self.logger.debug('Packages environements not found')
            else:
                self.logger.debug('Found {} packages environments'.format(found_pkg_environments))

        return result

    def create_environment(self, confs_path, parsed_args):
        return Environment(
            confs_path=confs_path,
            chroot=self.chroot,
            logger=self.logger,
            skip_chown=parsed_args.skip_chown,
            rsa_login=parsed_args.rsa_login,
            rsa_private_key=parsed_args.rsa_private_key,
            rsa_agent_key_hash=parsed_args.rsa_agent_key_hash,
            quiet=parsed_args.quiet,
            debug=parsed_args.debug,
            config_files=parsed_args.config_files,
            testing=parsed_args.testing,
            oauth_token=parsed_args.oauth_token,
        )

    def run(self, argv=None):
        argcomplete.autocomplete(self.args_parser)

        self.parsed_args = self.parse_args(argv)

        logging.basicConfig(
            level=logging.DEBUG if self.parsed_args.debug else logging.ERROR,
            format='%(levelname)s [%(name)s] %(message)s',
        )
        logging.captureWarnings(True)
        if self.parsed_args.quiet:
            logging.disable(logging.CRITICAL)

        self.logger = logging.getLogger('yav-deploy')
        self.log_version()

        self.log_argv(argv or sys.argv)

        self.confs_path = os.path.expanduser(self.parsed_args.confs_path)
        self.chroot = os.path.expanduser(self.parsed_args.deploy_root)

        try:
            self.environments = self.find_environments(
                self.confs_path,
                self.parsed_args,
                skip_pkg=self.parsed_args.skip_pkg,
            )
        except Exception as e:
            self.logger.error(e, exc_info=True)
            self.exit(BAD_ARGUMENTS_STATUS)

        failed = False
        for env_ in self.environments:
            self.logger.debug(env_)
            if not env_.has_confs:
                self.logger.debug('Configuration files not found')
                continue

            env_ok = env_.run(
                force=self.parsed_args.force,
                valid_sections=self.parsed_args.sections,
                skip_post_update=self.parsed_args.skip_post_update,
                skip_pre_update=self.parsed_args.skip_pre_update,
            )
            if not env_ok:
                failed = True

        self.exit(HAS_ERRORS_STATUS if failed else OK_STATUS)
