# coding: utf-8

import re


ROLE_ACTIONS = ('assign', 'revoke')
ROLE_TYPES = ('owner', 'reader', 'appender')
ROLE_SOURCES = ('user', 'abc', 'staff')
DEFAULT_ROLE_SOURCE = 'user'

ROLE_RE = re.compile(
    r'''
        ^
        (?P<action>[+-])?(?P<role_type>[^\s:]+?):((?P<source>[^\s:]+?):)?(?P<entity_id>[^\s:]+)
        (?:
            (?:\:role\:(?P<role_id>[^\s:]+))
            |
            (?:\:(?:scope\:)?(?P<scope>[^\s:]+))
        )?
        $
    ''',
    re.I | re.X,
)
UID_RE = re.compile(r'^\d+$')


def format_role(role):
    description = None
    if 'uid' in role:
        source = 'user'
        entity_id = role.get('login') or role['uid']
    elif 'abc_id' in role:
        source = 'abc'
        entity_id = role.get('abc_slug') or role['abc_id']

        scope_slug = role.get('abc_scope')
        role_id = role.get('abc_role')
        if scope_slug:
            entity_id = u'{}:scope:{}'.format(entity_id, scope_slug)
        elif role_id:
            entity_id = u'{}:role:{}'.format(entity_id, role_id)

        description = role.get('abc_name') or ''
        scope_name = role.get('abc_scope_name')
        role_name = role.get('abc_role_name')
        if scope_name is not None:
            description = u'{}. Scope: {}'.format(
                description.rstrip('.'),
                scope_name,
            )
        elif role_name is not None:
            description = u'{}. Role: {}'.format(
                description.rstrip('.'),
                role_name,
            )

    elif 'staff_id' in role:
        source = 'staff'
        entity_id = role.get('staff_slug') or role['staff_id']
        description = role.get('staff_name')
    else:
        source = 'unknown'
        entity_id = 'unknown'

    return u'{slug}:{source}:{entity_id}{description}'.format(
        slug=role['role_slug'].lower(),
        source=source,
        entity_id=entity_id,
        description=u' ({})'.format(description) if description else '',
    )


class RoleAction(object):
    def __init__(self, action, role_type, source, entity_id, scope=None, role_id=None):
        if action.lower() in ROLE_ACTIONS:
            self.action = action.lower()
        else:
            raise ValueError('"{}" is an unknown roles action value'.format(action))

        if role_type.lower() in ROLE_TYPES:
            self.role_type = role_type.lower()
        else:
            raise ValueError('"{}" is an unknown roles type value'.format(role_type))

        if source.lower() in ROLE_SOURCES:
            self.source = source.lower()
        else:
            raise ValueError('"{}" is an unknown roles source value'.format(source))

        if entity_id:
            self.entity_id = entity_id
        else:
            raise ValueError('An id value is undefined')

        self.scope = scope

        self.role_id = role_id
        if self.role_id is not None and not UID_RE.search(self.role_id):
            raise ValueError('The role_id value ("{}") can only contain digits'.format(self.role_id))

    def __str__(self):
        return '{action}{type}:{source}:{entity_id}{scope}{role_id}'.format(
            action='-' if self.action == 'revoke' else '+',
            type=self.role_type,
            source=self.source,
            entity_id=self.entity_id,
            scope=':scope:{}'.format(self.scope) if self.scope is not None else '',
            role_id=':role:{}'.format(self.role_id) if self.role_id is not None else '',
        )

    def __repr__(self):
        return '<RoleAction("{}", "{}", "{}", "{}"{}{})>'.format(
            self.action,
            self.role_type,
            self.source,
            self.entity_id,
            ', scope="{}"'.format(self.scope) if self.scope is not None else '',
            ', role_id="{}"'.format(self.role_id) if self.role_id is not None else '',
        )

    def as_method_params(self):
        result = dict()
        if self.source == 'user':
            if UID_RE.search(self.entity_id):
                result['uid'] = self.entity_id
            else:
                result['login'] = self.entity_id
        elif self.source == 'staff':
            result['staff_id'] = self.entity_id
        elif self.source == 'abc':
            result['abc_id'] = self.entity_id
            if self.scope is not None:
                result['abc_scope'] = self.scope
            if self.role_id is not None:
                result['abc_role_id'] = self.role_id
        return result

    @staticmethod
    def from_string(str_):
        """
        [+-](reader|owner):(user|staff|abc):id(:scope:scope_slug|:role:role_id)
        +reader:login
        +reader:user:login
        +reader:staff:staff_group_id
        +reader:abc:abc_group_id
        +reader:abc:abc_group_id:abc_scope_slug
        +reader:abc:abc_group_id:scope:abc_scope_slug
        +reader:abc:abc_group_id:role:abc_role_id
        """
        str_ = str_.strip()
        parsed_role = ROLE_RE.match(str_)
        if not parsed_role:
            raise ValueError('"{}" is an invalid roles value'.format(str_))
        return RoleAction(
            'revoke' if parsed_role.group('action') == '-' else 'assign',
            parsed_role.group('role_type'),
            parsed_role.group('source') or DEFAULT_ROLE_SOURCE,
            parsed_role.group('entity_id').strip(),
            parsed_role.group('scope') or None,
            parsed_role.group('role_id') or None,
        )
