import 'promise-polyfill/src/polyfill';

import React from 'react';
import ReactDOM from 'react-dom';
import { setOrigin, addEvent, receiveMessage, sendMessageToParent, getCookie, addClassNames } from '@client/util';
import { messages } from '@client/constants';
import { metrika } from '@common/metrika';
import { frameManager } from '@common/browser-manager/frame';
import { register as registerBrowserManager, getBrowserManager } from '@common/browser-manager';
import { AppDataContextValue } from '@common/data/AppDataContext';

const APP_DATA = {
    models: {
        emails: window.__USER__?.emails ?? [],
        phones: window.__USER__?.phones ?? [],
        addresses: window.__USER__?.addresses ?? [],
    },
    profiles: window.__PROFILES__ ?? [],
    locals: window.__LOCALS__,
    meta: window.__META__,
    initialError: window.__INITIAL_ERRORS__,
    constants: window.__CONSTANTS__,
    users: window.__USERS__ ?? [],
    provider: window.__PROVIDER__ ?? {},
};

export const commonRender = (AppComponent: React.FC<{ initialData: AppDataContextValue }>) => {
    registerBrowserManager(frameManager);
    setOrigin(APP_DATA.meta.targetOrigin);

    const getErrorState = () => {
        if (APP_DATA.initialError.user) {
            return { type: 'user', code: APP_DATA.initialError.user };
        }
        if (APP_DATA.initialError.users) {
            return { type: 'users', code: APP_DATA.initialError.users };
        }
        if (APP_DATA.initialError.provider) {
            return { type: 'provider', code: APP_DATA.initialError.provider };
        }
        return null;
    };

    let wasAppRendered = false;
    const renderApp = () => {
        if (
            wasAppRendered ||
            (!APP_DATA.meta.whitelist.length && !['suggest', 'provider'].includes(APP_DATA.constants.type))
        ) {
            return;
        }

        const errorState = getErrorState();

        if (errorState) {
            return sendMessageToParent({
                cause: 'autofill',
                type: messages.error,
                payload: errorState,
            });
        }

        wasAppRendered = true;

        ReactDOM.render(<AppComponent initialData={APP_DATA} />, document.getElementById('root'));
        addClassNames(document.body, ['visible']);

        sendMessageToParent({
            cause: 'sdk',
            type: messages.rendered,
        });
    };

    receiveMessage(APP_DATA.meta.targetOrigin, ({ type }) => {
        const {
            meta: { controlSum },
        } = APP_DATA;
        const errorState = getErrorState();
        const cookieState = (getCookie('controlSum') !== controlSum && { type: 'env', code: 'blocked' }) || null;
        const error = cookieState || errorState;

        if (error && error.code === 'NEED_RESET') {
            error.code = 'no_data';
        }

        if (type === 'open') {
            renderApp();
        }

        if (type === 'ping') {
            sendMessageToParent({
                cause: 'autofill',
                type: messages.loaded,
                payload: !error ? undefined : { error },
            });
        }
    });

    addEvent(window, 'keydown', e => {
        if (e.which === 27 || e.keyCode === 27 || e.code === 'Escape') {
            metrika.params({
                close: 'escape',
            });
            getBrowserManager().onClose();
        }
    });

    addEvent(window, 'DOMContentLoaded', () => {
        const {
            constants: { type, passportAuthUrl, passportAuthUpdateUrl },
            meta: { controlSum },
        } = APP_DATA;
        const errorState = getErrorState();
        const cookieState = (getCookie('controlSum') !== controlSum && { type: 'env', code: 'blocked' }) || null;
        const error = cookieState || errorState;

        if (type === 'popup') {
            sendMessageToParent({
                cause: 'autofill',
                type: messages.loaded,
                payload: !errorState ? undefined : { error },
            });
            if (error && error.code === 'NEED_RESET') {
                location.href = passportAuthUpdateUrl;
            }

            if (errorState) {
                location.href = passportAuthUrl;
            }
        } else if (type === 'provider') {
            sendMessageToParent({
                cause: 'provider',
                type: messages.loaded,
                payload: !errorState ? undefined : { error },
            });
        }
    });

    if (window.__CONSTANTS__.dev) {
        addEvent(window, 'message', e => {
            // eslint-disable-next-line no-console
            console.log('message: ', e.data);
        });
    }

    if (~location.search.indexOf('open=1')) {
        renderApp();
    }
};
