import { includesInArray, assign } from '@common/util';
import { messages } from './constants';
import { getCookie, sendMessage, setCookie } from './util';
import { loadIframe, loadPopup, Control } from './loadIframe';
import { getMetrika, createMetrika } from './metrika';
import { IframeType } from './types';
import { getStubControl, StubControl } from './getStubControl';

export const version: string = VERSION || '';

let status = '';
interface statusesMap {
    [key: string]: {
        timeout: number;
    };
}
const statuses: statusesMap = {
    unknown: {
        timeout: 20_000,
    },
    long_time_no_see: {
        timeout: 20_000,
    },
    denied: {
        timeout: 1_000,
    },
    timeout: {
        timeout: 300_000,
    },
    in_progress: {
        timeout: 0,
    },
    blocked: {
        timeout: 300_000,
    },
    not_available: {
        timeout: 10_800_000,
    },
    no_user: {
        timeout: 0,
    },
    no_docs: {
        timeout: 0,
    },
    no_process_name: {
        timeout: 0,
    },
    bad_origin: {
        timeout: 999_999_999_999,
    },
    no_data: {
        timeout: 300_000,
    },
    ready: {
        timeout: 300_000,
    },
};

let config: Config = {
    clientID: '',
    SRC: '',
    SRC2: '',
    targetOrigin: '',
    tokenPageOrigin: '',
    fields: [],
    theme: 'light',
    parentId: '',
    suggestView: 'default',
};

export const getConfig = () => config;

export const setConfig = (opts: Config) => {
    config = {
        ...config,
        ...opts,
    };
    return getConfig();
};

const filterAnswer = (data: MessageData, fields: RequestFields): Answer => {
    const result: Result = assign({}, data.payload);
    const answer: Answer = {};

    if (!result) {
        return answer;
    }

    fields.forEach((key: RequestField) => {
        switch (key) {
            case 'name':
                answer.firstName = result.firstName;
                answer.lastName = result.lastName;
                break;
            case 'address':
                answer.streetAddress = result.address;
                break;
            case 'email':
                answer.email = result[key];
                break;
            case 'phone':
                answer.phoneNumber = result[key];
                break;
        }
    });

    return answer;
};

export const getStatus = () => {
    return status || (status = getCookie());
};

export const setStatus = (newStatus = 'no_data') => {
    const { timeout: timeoutMs } = statuses[newStatus];
    let oldStatus = getStatus();

    if (oldStatus !== newStatus && timeoutMs) {
        setCookie({
            value: newStatus,
            timeoutMs,
        });
    }

    status = newStatus;
    return getStatus();
};

export const envCheck = () =>
    new Promise<EnvCheckResult>((resolve, reject) => {
        let cookieStatus = getCookie();

        if (!cookieStatus) {
            setCookie({
                value: 'long_time_no_see',
                timeoutMs: statuses.long_time_no_see.timeout,
            });

            cookieStatus = getCookie();
        }

        switch (cookieStatus) {
            case 'in_progress':
            case 'timeout':
            case 'no_data':
            case 'not_available':
            case 'blocked':
            case 'denied':
                return reject({
                    status: 'error',
                    code: cookieStatus,
                });
            case 'long_time_no_see':
                return resolve({
                    error: null,
                });
            default:
                return resolve({
                    status: 'error',
                    code: 'no_cookie',
                });
        }
    });

export const apiCheck = (type: IframeType) =>
    new Promise<ApiCheckResult>((resolve, reject) =>
        loadIframe(type)
            .then((control: Control) => {
                setTimeout(() => {
                    reject({
                        status: 'error',
                        code: 'not_available',
                    });
                }, 20000);

                control
                    .send({
                        cause: 'sdk',
                        type: 'ping',
                    })
                    .then((data: MessageData) => {
                        const payload: Payload | undefined = data.payload;
                        const type: string = data.type;

                        if (type === 'loaded') {
                            if (typeof payload === 'undefined') {
                                resolve(control);
                            } else {
                                reject({
                                    status: 'error',
                                    code: payload.error?.code,
                                });
                            }
                        }
                    });
            })
            .catch(error => {
                reject({
                    status: 'error',
                    code: error.message,
                });
            }),
    );

const ANIM_TIMEOUT = 300;

export const showIframeAutofill: () => Promise<Answer | AnswerWithError> = () =>
    new Promise((resolve, reject) => {
        loadIframe('autofill').then((control: Control) => {
            control.subscribe((data: MessageData) => {
                if (includesInArray([messages.accepted, messages.denied], data.type)) {
                    setTimeout(() => control.destroy(), ANIM_TIMEOUT);

                    if (data.type === messages.accepted) {
                        const result = filterAnswer(data, getConfig().fields);
                        resolve(result);
                    } else {
                        setStatus(data.type);
                        reject(data.type);
                    }
                    return;
                }
            });

            control.show();
        });
    });

const sendSuggestMetric = (metricParams: object | string) =>
    getMetrika()?.params({ 'Саджест Авторизации': metricParams });

export const showIframeSuggest: () => Promise<object | undefined> = () =>
    loadIframe('suggest').then((control: Control) => {
        return new Promise((resolve, reject) => {
            const { tokenPageOrigin } = getConfig();

            control.subscribe((data: MessageData, e: MessageEvent) => {
                switch (data.type) {
                    case messages.height:
                        const { height } = data.payload as { height?: string };

                        return height && control.setHeight?.(height);
                    case messages.metric:
                        const { metrikaId, encodedBoxes } = data.payload as {
                            metrikaId?: string | null;
                            encodedBoxes?: string | null;
                        };
                        return createMetrika({ metrikaId, encodedBoxes });
                    case messages.token:
                        setTimeout(() => control.destroy(), ANIM_TIMEOUT);
                        if (tokenPageOrigin && e.source) {
                            sendMessage(
                                { cause: 'token', payload: {}, type: messages.close },
                                tokenPageOrigin,
                                e.source as Window,
                            );
                        }
                        sendSuggestMetric('token_sucсses');
                        return resolve(data.payload);
                    case messages.denied:
                        setTimeout(() => control.destroy(), ANIM_TIMEOUT);
                        sendSuggestMetric({ token_fail: data.type });
                        setStatus(data.type);
                        return reject(data.type);
                }
            });

            control.show();

            control.send({ type: messages.metric, cause: 'suggest' });
        });
    });

export const showSuggestButtonStub: () => Promise<object | undefined> = () =>
    getStubControl('suggest').then((control: StubControl) => {
        return new Promise((resolve, reject) => {
            const { tokenPageOrigin } = getConfig();

            control.subscribe((data: MessageData, e: MessageEvent) => {
                switch (data.type) {
                    case messages.token:
                        setTimeout(() => control.destroy(), ANIM_TIMEOUT);
                        if (tokenPageOrigin && e.source) {
                            sendMessage(
                                { cause: 'token', payload: {}, type: messages.close },
                                tokenPageOrigin,
                                e.source as Window,
                            );
                        }
                        sendSuggestMetric('token_sucсses');
                        return resolve(data.payload);
                    case messages.denied:
                        setTimeout(() => control.destroy(), ANIM_TIMEOUT);
                        sendSuggestMetric({ token_fail: data.type });
                        setStatus(data.type);
                        return reject(data.type);
                }
            });

            control.show();

            sendSuggestMetric('show_button_stub');
        });
});

export const showIframeProvider: () => Promise<object | undefined> = () =>
    loadIframe('provider').then((control: Control) => {
        return new Promise((resolve, reject) => {
            control.subscribe((data: MessageData) => {
                switch (data.type) {
                    case messages.metric:
                        const { metrikaId, encodedBoxes } = data.payload as {
                            metrikaId?: string | null;
                            encodedBoxes?: string | null;
                        };
                        return createMetrika({ metrikaId, encodedBoxes });
                    case messages.accepted:
                        setTimeout(() => control.destroy(), ANIM_TIMEOUT);
                        return resolve({ status: 'ok', action: 'accepted', payload: { kek: true } });
                    case messages.close:
                        setTimeout(() => control.destroy(), ANIM_TIMEOUT);
                        return resolve({ status: 'ok', action: 'close' });
                    case messages.denied:
                        setTimeout(() => control.destroy(), ANIM_TIMEOUT);
                        return resolve({ status: 'ok', action: 'denied' });
                    case messages.closeBySdk:
                        setTimeout(() => control.destroy(), ANIM_TIMEOUT);
                        return resolve({ status: 'ok', action: 'close' });
                    case messages.error:
                        setTimeout(() => control.destroy(), ANIM_TIMEOUT);
                        return reject({ status: 'error', payload: data.payload });
                }
            });

            control.show();

            control.send({ type: messages.metric, cause: 'provider' });
        });
    });

export const showIframeSocial: () => Promise<object | undefined> = () =>
    loadIframe('social').then((control: Control) => {
        const { targetOrigin } = getConfig();

        return new Promise(resolve => {
            control.subscribe((data: MessageData, e: MessageEvent) => {
                switch (data.type) {
                    case messages.social:
                        if (targetOrigin && e.source) {
                            sendMessage(
                                { cause: 'social', payload: {}, type: messages.close },
                                targetOrigin,
                                e.source as Window,
                            );
                        }
                        return resolve(data.payload);
                }
            });

            control.show();
        });
    });

export const showPopup: () => Promise<Answer | AnswerWithError> = () =>
    new Promise((resolve, reject) => {
        loadPopup().then((control: Control) => {
            control.subscribe((data: MessageData) => {
                const payload: Payload | undefined = data.payload;
                const type: string = data.type;

                if (type === 'loaded') {
                    if (typeof payload === 'undefined') {
                        control.show();
                    } else {
                        reject({
                            status: 'error',
                            code: payload.error?.code,
                        });
                    }
                }

                if (includesInArray([messages.accepted, messages.denied], data.type)) {
                    setTimeout(() => control.destroy(), ANIM_TIMEOUT);

                    if (data.type === messages.accepted) {
                        const result = filterAnswer(data, getConfig().fields);
                        resolve(result);
                    } else {
                        setStatus(data.type);
                        reject(data.type);
                    }
                    return;
                }
            });
            control.show();
        });
    });
