import { sendMessage, receiveMessage } from './util';
import { getConfig } from './common';
import { IframeType } from './types';

export type Control = {
    elem: HTMLIFrameElement | Window | null;
    setHeight?: (height: string) => void;
    send: (data: MessageData) => Promise<MessageData>;
    subscribe: (handler: (data: MessageData, e: MessageEvent) => void) => void;
    destroy: () => void;
    show: () => void;
};

let control: Promise<Control>;

const popupWindow = (url: string, windowName: string, win: Window, w: number, h: number) => {
    if (!win.top) {
        return null;
    }
    const y = win.top.outerHeight / 2 + win.top.screenY - h / 2;
    const x = win.top.outerWidth / 2 + win.top.screenX - w / 2;

    return win.open(
        url,
        windowName,
        `toolbar=no, location=no, directories=no, status=no, menubar=no, scrollbars=no, resizable=no, copyhistory=no, width=${w}, height=${h}, top=${y}, left=${x}`,
    );
};

export const loadIframe: (type: IframeType) => Promise<Control> = type => {
    const { targetOrigin, tokenPageOrigin = '', theme, parentId = '', suggestView = 'default' } = getConfig();

    return (
        control ||
        (control = new Promise((resolve, reject) => {
            const iframe: HTMLIFrameElement = document.createElement('iframe');
            let loaded = false;
            let loading = false;
            let load: ((onReload: () => void) => void) | null = null;
            let onReLoad = () => {};

            iframe.id = 'iframe';
            iframe.style.position = 'fixed';
            iframe.style.right = '0';
            iframe.style.top = '0';
            iframe.style.border = '0';
            iframe.style.zIndex = '9999'; // TODO
            iframe.style.display = 'none'; // TODO

            if (['autofill', 'provider'].includes(type)) {
                iframe.width = '100%';
                iframe.height = '100%';
            } else if (type === 'social') {
                if (window.screen.width > 620) {
                    iframe.width = '600';
                } else if (window.screen.width > 420) {
                    iframe.width = '400';
                } else {
                    iframe.width = '100%';
                }
            } else if (['suggest'].includes(type)) {
                if (['button'].includes(suggestView)) {
                    iframe.width = '100%';
                    iframe.style.position = 'relative';
                } else {
                    iframe.width = window.screen.width > 400 ? '380' : '100%';
                }
            }

            const subscribe = (handler: (data: MessageData, e: MessageEvent) => void) => {
                const handlers: (() => void)[] = [];
                handlers.push(receiveMessage(targetOrigin, handler));
                if (type === 'suggest' && tokenPageOrigin) {
                    handlers.push(receiveMessage(tokenPageOrigin, handler));
                }

                return () => handlers.forEach(handler => handler());
            };
            const send = (dataToSend: MessageData) =>
                new Promise<MessageData>(resolve => {
                    const unsubscribe = subscribe(data => {
                        if (data.cause === 'autofill') {
                            resolve(data);
                            unsubscribe();
                        }
                    });

                    sendMessage(dataToSend, targetOrigin, iframe.contentWindow);
                });

            const destroy = () => {
                iframe.style.display = 'none';
                iframe.parentNode?.removeChild(iframe);
                loaded = false;
            };

            const show = () => {
                iframe.style.display = 'block';

                if (loaded) {
                    send({
                        cause: 'sdk',
                        type: 'open',
                        payload: {
                            theme,
                        },
                    });
                } else {
                    // eslint-disable-next-line no-use-before-define
                    load?.(onReLoad);
                }
            };

            const setHeight = (height: string) => {
                iframe.height = height;
            };

            const onLoad: () => void = () => {
                iframe.removeEventListener('load', onLoad);
                loaded = true;
                loading = false;

                resolve({
                    elem: iframe,
                    setHeight,
                    send,
                    subscribe,
                    destroy,
                    show,
                });
            };

            onReLoad = () => {
                iframe.removeEventListener('load', onReLoad);
                loaded = true;
                loading = false;
                show();
            };

            load = (onLoad: () => void) => {
                if (loading) {
                    return;
                }

                loading = true;
                const { SRC } = getConfig();
                iframe.src = SRC;
                if (parentId && document.getElementById(parentId)) {
                    document.getElementById(parentId)?.appendChild(iframe);
                } else {
                    document.body.appendChild(iframe);
                }
                iframe.addEventListener('load', onLoad);
            };

            load(onLoad);

            setTimeout(() => {
                if (!loaded) {
                    destroy();
                }
                reject(new Error('timeout'));
            }, 20000);
        }))
    );
};

export const loadPopup: () => Promise<Control> = () => {
    const { SRC2: SRC, targetOrigin, theme } = getConfig();

    return new Promise((resolve, reject) => {
        let popup: Window | null;
        let loaded = false;
        let loading = false;

        const subscribe = (handler: (data: MessageData, e: MessageEvent) => void) =>
            receiveMessage(targetOrigin, handler);
        const send = (dataToSend: MessageData) =>
            new Promise<MessageData>(resolve => {
                const unsubscribe = subscribe(data => {
                    if (data.cause === 'autofill') {
                        resolve(data);
                        unsubscribe();
                    }
                });

                sendMessage(dataToSend, targetOrigin, popup);
            });

        const destroy = () => {
            popup && popup.close();
            loaded = false;
        };

        const show = () => {
            send({
                cause: 'sdk',
                type: 'open',
                payload: {
                    theme,
                },
            });
        };

        const onLoad = () => {
            loaded = true;
            loading = false;

            resolve({
                elem: popup,
                send,
                subscribe,
                destroy,
                show,
            });
        };

        const load = (onLoad: () => void) => {
            if (loading) {
                return;
            }

            loading = true;

            subscribe(data => {
                if (data.type === 'loaded') {
                    onLoad();
                }
            });

            popup = popupWindow(SRC, 'popup', window, 500, 750);
        };

        load(onLoad);

        setTimeout(() => {
            if (!loaded) {
                destroy();
            }
            reject(new Error('timeout'));
        }, 20000);
    });
};
