import { isOldIE } from './constants';

let origin = '';

const csrf = window.__CSRF__;
const { uid } = window.__USER__ || { uid: null };

type EventListenerMethod = <K extends keyof HTMLElementEventMap>(
    elem: Window | Document | HTMLElement | null,
    type: K,
    listener: (this: HTMLElement, ev: HTMLElementEventMap[K]) => unknown,
) => void;

export const getScrollTop = () =>
    typeof pageYOffset !== 'undefined' ? pageYOffset : (document.documentElement || document.body).scrollTop;

export const addEvent: EventListenerMethod = (elem, type, listener) => {
    if (!elem) {
        return;
    }
    if ('addEventListener' in elem) {
        // @ts-ignore
        elem.addEventListener(type, listener, false);
    } else {
        // @ts-ignore
        elem.attachEvent('on' + type, listener);
    }
};

export const removeEvent: EventListenerMethod = (elem, type, listener) => {
    if (!elem) {
        return;
    }
    if ('removeEventListener' in elem) {
        // @ts-ignore
        elem.removeEventListener(type, listener, false);
    } else {
        // @ts-ignore
        elem.detachEvent('on' + type, listener);
    }
};

export const sendMessage = (message: MessageData, targetOrigin: string, to: Window | null = window) => {
    try {
        targetOrigin && to?.postMessage(isOldIE ? JSON.stringify(message) : message, targetOrigin);
    } catch (e) {
        /* EMPTY */
    }
};

export const setOrigin = (newOrigin: string) => (origin = newOrigin);
export const sendMessageToParent = (message: MessageData) => {
    sendMessage(message, origin, window.opener || parent);
};

export const receiveMessage = (
    from: string,
    onReceive: (data: MessageData, originEvent: MessageEvent) => unknown,
): (() => void) => {
    const onMessage = (e: MessageEvent) => {
        if (e.origin === from || e.origin === window.origin) {
            const message = isOldIE ? JSON.parse(e.data) : e.data;
            onReceive(message, e);
        }
    };

    addEvent(window, 'message', onMessage);

    return () => removeEvent(window, 'message', onMessage);
};

export const debounce = <T extends (...args: ANY) => unknown>(fn: T, delay: number) => {
    let timer: ReturnType<typeof setTimeout>;

    return (...args: Parameters<T>) => {
        clearTimeout(timer);
        timer = setTimeout(() => {
            // @ts-ignore
            fn(...args);
        }, delay);
    };
};

export const entries = <K extends string, V>(obj: Partial<Record<K, V>>) => {
    const result: [K, V | undefined][] = [];

    for (const k in obj) {
        result.push([k, obj[k]]);
    }

    return result;
};

export const values = <V>(obj: Record<string, V>) => {
    const result: V[] = [];

    for (const k in obj) {
        result.push(obj[k]);
    }

    return result;
};

export const fetch = <
    ResBody = ANY,
    ReqBody extends Record<string, string | number | undefined> = Record<string, string | number>
>(
    url: string,
    params: ReqBody,
    {
        method = 'POST',
        onSuccess,
        onError,
        always,
    }: {
        method?: 'GET' | 'POST';
        onSuccess: (res: ResBody) => unknown;
        onError?: (err?: Error | ResBody, statusCode?: number) => unknown;
        always?: () => unknown;
    },
) => {
    const xhr = new window.XMLHttpRequest();
    const queryParams = !params
        ? ''
        : '?' +
          entries(params)
              .map(([key, value = '']) => `${encodeURIComponent(key)}=${encodeURIComponent(value)}`)
              .join('&');

    xhr.open(method, url + (method === 'GET' ? queryParams : ''), true);
    xhr.withCredentials = true;

    if (method === 'POST') {
        xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
    }

    if (url[0] === '/') {
        uid && xhr.setRequestHeader('x-visible-uid', uid);
        xhr.setRequestHeader('x-csrf-token', csrf);
    }

    xhr.send(method === 'GET' || !queryParams ? undefined : queryParams.slice(1));

    xhr.addEventListener('readystatechange', () => {
        if (xhr.readyState !== 4) {
            return;
        }
        try {
            const json = JSON.parse(xhr.responseText);
            xhr.status === 200 ? onSuccess(json) : onError?.(json, xhr.status);
        } catch (_) {
            onError?.(new Error('invalid response'));
        }
        always?.();
    });
    xhr.addEventListener('error', () => {
        onError?.();
        always?.();
    });
};

export const getCookie = (name = '_ru_yandex_autofill') =>
    window.document.cookie.replace(new RegExp('(?:(?:^|.*;\\s*)' + name + '\\s*=\\s*([^;]*).*$)|^.*$'), '$1');

export const setCookie = ({ name = '_ru_yandex_autofill', value = '', timeoutMs = 5 * 60 * 1000 }) => {
    const expires = new Date(new Date().getTime() + timeoutMs);

    document.cookie = name + '=' + value + ';Expires=' + expires.toUTCString();
};

export const clearCookie = (name = '_ru_yandex_autofill') => {
    setCookie({
        name: name,
        value: '',
        timeoutMs: 0,
    });
};

export const hasClassName = (el: HTMLElement, name: string) => {
    return el.classList ? el.classList.contains(name) : !!~el.className.split(' ').indexOf(name);
};

export const addClassNames = (el: HTMLElement, names: string[]) => {
    const filtered = names.filter(name => !hasClassName(el, name));
    if (filtered.length) {
        el.className = `${el.className} ${filtered.join(' ')}`;
    }
};

export const removeClassNames = (el: HTMLElement, names: string[]) => {
    el.className = el.className
        .split(' ')
        .filter(c => !~names.indexOf(c))
        .join(' ');
};
