import { messages } from '@client/constants';
import { removeClassNames, sendMessageToParent, entries, fetch } from '@client/util';
import { metrika } from '@common/metrika';
import { validate, createId } from '@src/util';
import { prepareFields } from './util';

import type { APIBody } from '@src/types';
import type { UserProfile, Field } from '@src/types';
import type { BrowserManager, ValidationErrors } from './types';

const hideForm = () => removeClassNames(document.body, ['visible']);

const onClose = () => {
    hideForm();
    sendMessageToParent({ cause: 'autofill', type: messages.denied });
};

const onAccepted: BrowserManager['onAccepted'] = (data: UserProfile, options = { needValidate: false }) => {
    const { needValidate } = options;
    const { middleName, fullName, ...commonData } = data;
    const filledFields = prepareFields(commonData);

    const sendData = () => {
        metrika.params({ fill: 'success' });
        hideForm();

        return Promise.resolve(
            sendMessageToParent({
                cause: 'autofill',
                type: messages.accepted,
                payload: filledFields,
            }),
        );
    };

    if (!needValidate) {
        return sendData();
    }

    const profiles = window.__PROFILES__;
    const internalError = window.__LOCALS__.errors.internal;

    // Ищем новые контакты, которые ввел пользователь для сохранения через апи
    const { newFields, validationErrors } = entries(filledFields).reduce<{
        newFields: Partial<Record<Field, string>> | null;
        validationErrors: ValidationErrors | null;
    }>(
        (acc, [field, value]) => {
            if (!value || profiles.some(profile => profile[field] === value)) {
                return acc;
            }

            const error = validate[field](value);

            if (error) {
                acc.validationErrors = acc.validationErrors || {};
                acc.validationErrors[field] = error;
            }

            acc.newFields = acc.newFields || {};
            acc.newFields[field] = value;

            return acc;
        },
        { newFields: null, validationErrors: null },
    );

    // Если есть ошибки валидации, прерываем заполнение
    if (validationErrors) {
        metrika.params({ fill: 'validation error' });
        return Promise.resolve({ validationErrors });
    }

    // Если нет новых полей и нет ошибок, то просто выполняем заполнение
    if (!newFields) {
        return sendData();
    }

    return new Promise(resolve => {
        fetch<APIBody['save']['res'], APIBody['save']['req']>(
            '/api/save',
            { id: createId(), ...newFields },
            {
                onSuccess: ({ status, errors }) => {
                    if (status === 'ok') {
                        resolve(sendData());
                    } else if (errors) {
                        metrika.params({ fill: 'validation error' });
                        resolve({ validationErrors: errors });
                    } else {
                        metrika.params({ fill: 'request error' });
                        resolve({ extendedError: internalError });
                    }
                },
                onError: (error, statusCode) => {
                    if (statusCode === 401) {
                        sendMessageToParent({
                            cause: 'autofill',
                            type: messages.error,
                            payload: {
                                type: 'user',
                                code: 'no_auth',
                            },
                        });
                        metrika.params({ fill: 'account changed' });
                        resolve(onClose());
                    } else {
                        metrika.params({ fill: 'request error' });
                        resolve({ extendedError: internalError });
                    }
                },
            },
        );
    });
};

export const frameManager: BrowserManager = {
    onClose,
    onAccepted,
};
