import React, { useCallback, useContext, useState } from 'react';
import { ErrorContext } from '@common/components/Form/ErrorsContext';
import { ActiveScreenContext } from '@common/data/ActiveScreenContext';
import { metrika } from '@common/metrika';
import { getBrowserManager } from '@common/browser-manager';
import { AppDataContext } from '@common/data/AppDataContext';
import { ActiveProfileContext } from '@common/data/ActiveProfileContext';
import { BaseActionButton } from '@common/components/ActionButtons/BaseActionButton';
import { classNames } from '@common/util';

import styles from './fill-button.css';

type FillButtonProps = {
    needValidate: boolean;
    disabled?: boolean;
};

export const FillButton: React.FC<FillButtonProps> = ({ needValidate, disabled }) => {
    const [isSaving, setSaving] = useState(false);

    const { profile, index } = useContext(ActiveProfileContext);
    const { screen } = useContext(ActiveScreenContext);
    const {
        locals: {
            common,
            errors: { internal: internalError },
        },
    } = useContext(AppDataContext);
    const { setErrors } = useContext(ErrorContext);

    const onClick = useCallback(() => {
        if (isSaving) {
            return;
        }

        const clickTarget = screen === 'form' ? 'editForm' : 'profileList';
        const targetParams = index === -1 ? { isNewProfile: true } : { index: String(index + 1) };

        metrika.params({
            fill: {
                [clickTarget]: targetParams,
            },
        });

        setErrors({ validationErrors: null, internalError: null });

        setSaving(true);
        getBrowserManager()
            .onAccepted(profile, { needValidate })
            .then(result => {
                setSaving(false);

                if (!result) {
                    return;
                }
                const validationErrors = 'validationErrors' in result ? result.validationErrors : null;
                const internalError = 'extendedError' in result ? result.extendedError : null;

                setErrors({ validationErrors, internalError });
            })
            .catch(() => setErrors({ internalError, validationErrors: null }));
    }, [isSaving, needValidate, profile, screen, index, setErrors, internalError]);

    return (
        <BaseActionButton
            type="submit"
            dataT="fill-button"
            className={classNames(styles['fill-button'], disabled && styles.disabled)}
            disabled={disabled || isSaving}
            onClick={onClick}
        >
            {common.fill}
        </BaseActionButton>
    );
};
