import React, { useContext, useCallback, useState, useEffect, useRef } from 'react';
import { ErrorContext } from '@common/components/Form/ErrorsContext';
import { useAddressSuggest } from '@common/hooks/useAddressSuggest';
import { classNames } from '@common/util';
import { Input } from '@common/components/Input';
import { AppDataContext } from '@common/data/AppDataContext';
import { ActiveProfileContext } from '@common/data/ActiveProfileContext';

import styles from './address-suggest.css';

const calcMaxSuggestHeight = (node: HTMLElement) => {
    // Вычитаем небольшое расстояние от верхнего края, чтобы саджест не прибивался
    return node.getBoundingClientRect().bottom - 10;
};

type SuggestItemProps = {
    children: string;
    dataT: string;
    onSelect: (value: string) => void;
};

const SuggestItem: React.FC<SuggestItemProps> = ({ dataT, children, onSelect }) => {
    const onItemClick = useCallback(() => {
        onSelect(children);
    }, [onSelect, children]);

    return (
        <div data-t={dataT} className={styles['suggest-item']} onMouseDown={onItemClick}>
            {children}
        </div>
    );
};

type TooltipProps = {
    address: string;
    dataT: string;
    onAddressSelect: (address: string) => void;
};

const Tooltip: React.FC<TooltipProps> = ({ dataT, address, onAddressSelect }) => {
    const tooltipRef = useRef<HTMLDivElement>(null);
    const [tooltipStyles, setTooltipStyles] = useState<React.CSSProperties>({ maxHeight: 'auto' });

    const { items } = useAddressSuggest(address);

    useEffect(() => {
        if (tooltipRef.current) {
            setTooltipStyles({ maxHeight: `${calcMaxSuggestHeight(tooltipRef.current)}px` });
        }
    }, []);

    return (
        <div className={styles['tooltip-container']}>
            <div ref={tooltipRef} data-t={dataT} className={styles.tooltip} style={tooltipStyles}>
                {items.map((item, index) => (
                    <SuggestItem dataT={`${dataT}-item-${index}`} key={item} onSelect={onAddressSelect}>
                        {item}
                    </SuggestItem>
                ))}
            </div>
        </div>
    );
};

type AddressSuggestProps = {
    className?: string;
};

export const AddressSuggest: React.FC<AddressSuggestProps> = ({ className }) => {
    const {
        profile: { address = '' },
        updateProfile,
    } = useContext(ActiveProfileContext);

    const {
        locals: { extended, errors },
    } = useContext(AppDataContext);
    const { validationErrors, clearValidationError } = useContext(ErrorContext);

    const [isTooltipVisible, setTooltipVisible] = useState(false);

    const onAddressChange = useCallback(
        (address: string) => {
            updateProfile({ address });
        },
        [updateProfile],
    );
    const onFocus = useCallback(() => {
        clearValidationError('address');
        setTooltipVisible(true);
    }, [clearValidationError]);
    const onBlur = useCallback(() => setTooltipVisible(false), []);

    return (
        <div className={classNames(styles.suggest, className)}>
            <Input
                label={extended['new.address']}
                dataT="extended-new-address"
                value={address}
                onChange={onAddressChange}
                onFocus={onFocus}
                onBlur={onBlur}
                error={validationErrors && validationErrors.address && errors[validationErrors.address]}
            />
            {isTooltipVisible && (
                <Tooltip dataT="extended-address-suggest" address={address} onAddressSelect={onAddressChange} />
            )}
        </div>
    );
};

export default AddressSuggest;
