import { UserProfile } from '@src/types';
import React, { createContext, useState, useMemo, useCallback } from 'react';

import type { ValidationErrors } from '@common/browser-manager/types';

type ErrorsContextValue = {
    validationErrors: ValidationErrors | null;
    internalError: string | null;
    setErrors: (errors: { validationErrors: ValidationErrors | null; internalError: string | null }) => void;
    clearValidationError: (field: keyof UserProfile) => void;
    clearInternalError: () => void;
};

export const ErrorContext = createContext<ErrorsContextValue>({
    validationErrors: null,
    internalError: null,
    setErrors: () => {},
    clearValidationError: () => {},
    clearInternalError: () => {},
});

export const ErrorsProvider: React.FC = ({ children }) => {
    const [internalError, setInternalError] = useState<string | null>(null);
    const [validationErrors, setValidationErrors] = useState<ValidationErrors | null>(null);

    const setErrors = useCallback<ErrorsContextValue['setErrors']>(errors => {
        setValidationErrors(errors.validationErrors);
        setInternalError(errors.internalError);
    }, []);

    const clearValidationError = useCallback<ErrorsContextValue['clearValidationError']>(error => {
        setValidationErrors(errors => {
            if (!errors) {
                return errors;
            }

            const updatedErrors = { ...errors };
            delete updatedErrors[error];

            return updatedErrors;
        });
    }, []);

    const clearInternalError = useCallback(() => setInternalError(null), []);

    const value = useMemo<ErrorsContextValue>(() => {
        return {
            internalError,
            validationErrors,
            setErrors,
            clearValidationError,
            clearInternalError,
        };
    }, [internalError, validationErrors, setErrors, clearValidationError, clearInternalError]);

    return <ErrorContext.Provider value={value}>{children}</ErrorContext.Provider>;
};
