import React, { useContext, useCallback, useState, useEffect, useRef } from 'react';
import { ActiveProfileContext } from '@common/data/ActiveProfileContext';
import { Select, SelectOption } from '@common/components/Select';

import styles from './form-select.css';

type BaseFormSelectProps = {
    label: string;
    dataT: string;
    activeProfileOption: string;
    initialOptions: SelectOption[];
    onSelect: (option: SelectOption) => void;
};

export const BaseFormSelect: React.FC<BaseFormSelectProps> = props => {
    const { dataT, label, activeProfileOption, initialOptions, onSelect } = props;

    const { index: activeProfileIndex } = useContext(ActiveProfileContext);

    const [options, updateOptions] = useState(() => {
        return initialOptions.map<SelectOption>(option => ({ ...option }));
    });

    const changeCheckedOptionByIndex = useCallback((newIndex: number) => {
        updateOptions(options =>
            options.map((option, index) => ({
                ...option,
                checked: index === newIndex,
            })),
        );
    }, []);

    const changeCheckedOptionByActiveProfile = useCallback(
        (newActiveValue: string) => {
            updateOptions(options => {
                const { checkedIndex, updatedOptions } = options.reduce<{
                    checkedIndex: number;
                    updatedOptions: SelectOption[];
                }>(
                    ({ checkedIndex, updatedOptions }, option, curIndex) => {
                        checkedIndex = option.value === newActiveValue ? curIndex : checkedIndex;
                        updatedOptions.push({ ...option });

                        return { checkedIndex, updatedOptions };
                    },
                    { checkedIndex: -1, updatedOptions: [] },
                );

                if (updatedOptions.length) {
                    let index = checkedIndex === -1 ? 0 : checkedIndex;

                    updatedOptions[index].checked = true;
                    onSelect(updatedOptions[index]);
                }

                return updatedOptions;
            });
        },
        [onSelect],
    );

    const onOptionChange = useCallback(
        (_, newIndex: number) => {
            const newSelectedOption = options[newIndex];

            changeCheckedOptionByIndex(newIndex);
            onSelect(newSelectedOption);
        },
        [options, changeCheckedOptionByIndex, onSelect],
    );

    // Запоминаем значение опции в ref, чтобы не указывать ее в зависимости useEffect
    const storedActiveOption = useRef('');
    storedActiveOption.current = activeProfileOption ?? '';
    // Меняем выбранную опцию при смене активного профиля
    useEffect(() => {
        changeCheckedOptionByActiveProfile(storedActiveOption.current);
    }, [activeProfileIndex, changeCheckedOptionByActiveProfile]);

    return (
        <Select
            dataT={dataT}
            className={styles['form-select']}
            label={label}
            options={options}
            onOptionSelect={onOptionChange}
        />
    );
};
