import React, { useState, useContext, useCallback } from 'react';
import { ErrorContext } from '@common/components/Form/ErrorsContext';
import { AppDataContext } from '@common/data/AppDataContext';
import { Input } from '@common/components/Input';
import { ActiveProfileContext } from '@common/data/ActiveProfileContext';
import { BaseFormSelect } from '../BaseFormSelect';

import type { SelectOption } from '@common/components/Select';

import styles from './email-select.css';

export const FormEmailSelect: React.FC = () => {
    const { validationErrors, clearValidationError } = useContext(ErrorContext);

    const {
        profile: { email },
        updateProfile,
    } = useContext(ActiveProfileContext);
    const {
        models,
        locals: { common, extended, errors },
    } = useContext(AppDataContext);

    const [phoneOptions] = useState(() => {
        return models.emails
            .map<SelectOption>(email => ({ value: email, label: email }))
            .concat({ value: extended['add.email'], label: extended['new.email'], isNew: true });
    });
    const [isNewValueFieldVisible, setNewValueFieldVisible] = useState(false);

    const onSelect = useCallback(
        ({ isNew, value }: SelectOption) => {
            updateProfile({ email: isNew ? '' : value });
            setNewValueFieldVisible(Boolean(isNew));

            if (!isNew) {
                clearValidationError('email');
            }
        },
        [updateProfile, clearValidationError],
    );

    const onNewValueChange = useCallback(
        (email: string) => {
            updateProfile({ email });
        },
        [updateProfile],
    );

    return (
        <>
            <BaseFormSelect
                label={common.email}
                dataT="extended-email"
                activeProfileOption={email ?? ''}
                initialOptions={phoneOptions}
                onSelect={onSelect}
            />
            {isNewValueFieldVisible && (
                <Input
                    className={styles['new-email']}
                    label={extended['new.email']}
                    value={email ?? ''}
                    dataT="extended-new-email"
                    onChange={onNewValueChange}
                    error={validationErrors && validationErrors.email && errors[validationErrors.email]}
                />
            )}
        </>
    );
};
