import React, { useState, useContext, useCallback } from 'react';
import { ErrorContext } from '@common/components/Form/ErrorsContext';
import { AppDataContext } from '@common/data/AppDataContext';
import { Input } from '@common/components/Input';
import { ActiveProfileContext } from '@common/data/ActiveProfileContext';
import { BaseFormSelect } from '../BaseFormSelect';

import type { SelectOption } from '@common/components/Select';

import styles from './phone-select.css';

export const FormPhoneSelect: React.FC = () => {
    const { validationErrors, clearValidationError } = useContext(ErrorContext);

    const {
        profile: { phone },
        updateProfile,
    } = useContext(ActiveProfileContext);
    const {
        models,
        locals: { common, extended, errors },
    } = useContext(AppDataContext);

    const [phoneOptions] = useState(() => {
        return models.phones
            .map<SelectOption>(phone => ({ value: phone, label: phone }))
            .concat({ value: extended['add.phone'], label: extended['new.phone'], isNew: true });
    });
    const [isNewValueFieldVisible, setNewValueFieldVisible] = useState(false);

    const onSelect = useCallback(
        ({ isNew, value }: SelectOption) => {
            updateProfile({ phone: isNew ? '' : value });
            setNewValueFieldVisible(Boolean(isNew));

            if (!isNew) {
                clearValidationError('phone');
            }
        },
        [updateProfile, clearValidationError],
    );

    const onNewValueChange = useCallback(
        (phone: string) => {
            updateProfile({ phone });
        },
        [updateProfile],
    );

    return (
        <>
            <BaseFormSelect
                label={common.phone}
                dataT="extended-phone"
                activeProfileOption={phone ?? ''}
                initialOptions={phoneOptions}
                onSelect={onSelect}
            />
            {isNewValueFieldVisible && (
                <Input
                    className={styles['new-phone']}
                    label={extended['new.phone']}
                    dataT="extended-new-phone"
                    value={phone ?? ''}
                    onChange={onNewValueChange}
                    error={validationErrors && validationErrors.phone && errors[validationErrors.phone]}
                />
            )}
        </>
    );
};
