import { ErrorContext } from '@common/components/Form/ErrorsContext';
import React, { useContext, useCallback } from 'react';
import { ActiveProfileContext } from '@common/data/ActiveProfileContext';
import { AppDataContext } from '@common/data/AppDataContext';
import { Input, InputProps } from '@common/components/Input';

import type { UserProfile } from '@src/types';

import styles from './form-user-name.css';

type UserNameInputProps = Omit<InputProps, 'onChange'> & {
    field: keyof UserProfile;
    onChange: (field: keyof UserProfile, value: string) => void;
};

const UserNameInput: React.FC<UserNameInputProps> = ({ field, onChange, ...props }) => {
    const { validationErrors } = useContext(ErrorContext);
    const { errors } = useContext(AppDataContext).locals;

    const errorName = validationErrors && validationErrors[field];
    const error = errorName && errors[errorName];

    const onInputChange = useCallback(
        value => {
            onChange(field, value);
        },
        [field, onChange],
    );

    return <Input className={styles['user-name-item']} {...props} onChange={onInputChange} error={error} />;
};

export const FormUserName: React.FC = () => {
    const {
        profile: { firstName = '', lastName = '', middleName = '', fullName = '' },
        updateProfile,
    } = useContext(ActiveProfileContext);
    const {
        locals: { common },
        constants: { type },
    } = useContext(AppDataContext);

    const onFieldChange = useCallback(
        (field: keyof UserProfile, value: string) => {
            updateProfile({ [field]: value });
        },
        [updateProfile],
    );

    const isNameChunksEmpty = ![firstName, middleName, lastName].every(Boolean);
    const isFullNameEmpty = !fullName;

    // Для фрейма используется всегда два поля ввода - имя и фамилия
    if (type !== 'bro') {
        return (
            <>
                <UserNameInput
                    label={common['first-name']}
                    value={firstName}
                    field="firstName"
                    dataT="extended-first-name"
                    onChange={onFieldChange}
                />
                <UserNameInput
                    label={common['last-name']}
                    value={lastName}
                    field="lastName"
                    dataT="extended-last-name"
                    onChange={onFieldChange}
                />
            </>
        );
    }

    if (!isFullNameEmpty && isNameChunksEmpty) {
        return (
            <UserNameInput
                label={common['full-name']}
                value={fullName}
                field="fullName"
                dataT="extended-full-name"
                onChange={onFieldChange}
            />
        );
    }

    return (
        <>
            <UserNameInput
                label={common['first-name']}
                value={firstName}
                field="firstName"
                dataT="extended-first-name"
                onChange={onFieldChange}
            />
            <UserNameInput
                label={common['middle-name']}
                value={middleName}
                field="middleName"
                dataT="extended-middle-name"
                onChange={onFieldChange}
            />
            <UserNameInput
                label={common['last-name']}
                value={lastName}
                field="lastName"
                dataT="extended-last-name"
                onChange={onFieldChange}
            />
        </>
    );
};
