import React, { useCallback } from 'react';
import { classNames } from '@common/util';

import styles from './input.css';

type InputThemes = 'text' | 'select';

export type InputProps = Omit<React.InputHTMLAttributes<HTMLInputElement>, 'onChange'> & {
    label: string;
    dataT: string;
    error?: string | null;
    controlRef?: React.MutableRefObject<HTMLDivElement | null>;
    theme?: InputThemes;
    onChange?: (value: string) => void;
};

const themeClassNames: Record<InputThemes, string> = {
    text: styles.theme_text,
    select: styles.theme_select,
};

export const Input: React.FC<InputProps> = props => {
    const {
        dataT,
        className,
        controlRef,
        theme = 'text',
        disabled,
        label,
        value,
        onChange,
        onClick,
        onFocus,
        onBlur,
        error,
    } = props;
    const onInputChange = useCallback<React.ChangeEventHandler<HTMLInputElement>>(
        e => {
            onChange?.(e.target.value);
        },
        [onChange],
    );
    return (
        <label data-t={dataT} className={classNames(styles.label, themeClassNames[theme], className)}>
            <span className={styles['label-value']}>{label}</span>
            <span ref={controlRef} className={styles['label-content']} onClick={onClick}>
                <input
                    type="text"
                    autoComplete="chrome-off"
                    disabled={disabled}
                    className={classNames(styles.input, error && styles.input_error)}
                    value={value}
                    onChange={onInputChange}
                    onFocus={onFocus}
                    onBlur={onBlur}
                />
            </span>
            {error && (
                <span data-t="extended-error" className={styles.error}>
                    {error}
                </span>
            )}
        </label>
    );
};
