import React, { useContext, useState, useCallback, useEffect } from 'react';
import { classNames } from '@common/util';
import { SuggestUser as SuggestUserType } from '@src/types';
import { AppDataContext } from '@common/data/AppDataContext';
import ResizeObserver from 'resize-observer-polyfill';
import { getButtonView, getButtonTheme, calcCanShowLongText, setStyle } from './utils';

type PersonalButtonProps = {
    user: SuggestUserType;
    dataT: string;
    className?: string;
    onClick?: () => void;
    disabled?: boolean;
    type?: React.ButtonHTMLAttributes<HTMLButtonElement>['type'];
};

export const PersonalButton: React.FC<PersonalButtonProps> = ({ user: { avatar = '0/0-0', firstName = '', name = '', hasPlus = false } = {}, dataT, className, onClick, disabled, type = 'button' }) => {
    const {
        constants: {
            avatarTemplate = '',
            suggestButtonOptions: {
                buttonSize: size = 'm',
                buttonView: view = 'main',
                buttonTheme: theme = 'light',
                buttonBorderRadius: borderRadius = '0',
            } = {},
        } = {},
        locals: { suggest = {} } = {},
    } = useContext(AppDataContext);

    useEffect(() => {
        setStyle();
      }, []);

    const [canShowLongText, setCanShowLongText] = useState(false);

    const resizeObserver = React.useRef<ResizeObserver>(new ResizeObserver((entries:ResizeObserverEntry[]) => {
        const buttonWidth = entries[0]?.borderBoxSize[0]?.inlineSize;
        setCanShowLongText(calcCanShowLongText(size, buttonWidth));
    }));
    const buttonRef = useCallback((container: HTMLButtonElement) => {
        if (container !== null) {
            resizeObserver.current.observe(container);
        } else if (resizeObserver.current) {
                resizeObserver.current.disconnect();
        }
    }, [resizeObserver.current]);

    const currentName = firstName || name;
    const shortText = suggest['button.signin.short'];
    const longText = currentName ? suggest['button.signin.personal'].replace('%', currentName) : suggest['button.signin.default'];
    const buttonText = canShowLongText ? longText : shortText;
    const currentView = getButtonView(view);
    const currentTheme = getButtonTheme(theme);

    return (
        <button
            type={type}
            data-t={dataT}
            disabled={disabled}
            className={
                classNames(
                    'yaPersonalButton',
                    `yaPersonalButton_${size}`,
                    `yaPersonalButton_${currentView}`,
                    `yaPersonalButton_${currentTheme}`,
                    className,
                )}
            onClick={onClick}
            ref={buttonRef}
            style={{ borderRadius: `${borderRadius}px` }}
        >
            <div
                className={classNames('yaPersonalButtonLogo')}
                data-t="personal-button:logo"
            />
            <div className={'yaPersonalButtonInfo'} data-t="personal-button:info">
                <div className={'yaPersonalButtonText'} data-t="personal-button:text">
                    {buttonText}
                </div>
            </div>
            <div
                className={classNames('yaPersonalButtonAvatar', hasPlus && 'yaPersonalButtonAvatar_plus')}
                style={{
                    backgroundImage: `url("${avatarTemplate.replace('%avatar_id%', avatar).replace('%size%', '50')}")`,
                }}
                data-t="personal-button:avatar"
            />
        </button>
    );
};
