import React, { useState, useContext, useCallback, useEffect, useRef } from 'react';
import { metrika } from '@common/metrika';
import { classNames } from '@common/util';
import { AppDataContext } from '@common/data/AppDataContext';
import { ActiveScreenContext } from '@common/data/ActiveScreenContext';
import { ActiveProfileContext } from '@common/data/ActiveProfileContext';
import { DefaultButton } from '@common/components/ActionButtons/DefaultButton';
import { Profile } from '@common/components/Profile';

import styles from './profile-list.css';

// Запоминаем высоту блока и наличие клика по кнопке "Еще", чтобы сохранить вид списка пресетов
// после переходов между уровнями автофила
let wasMoreClicked = false;
let calculatedHeight: number;

type ButtonProps = {
    dataT: string;
    onClick: () => void;
};

const ActionButton: React.FC<ButtonProps> = ({ dataT, onClick, children }) => (
    <DefaultButton dataT={dataT} className={styles['action-button']} onClick={onClick}>
        {children}
    </DefaultButton>
);

export const ProfileList: React.FC = () => {
    const { index: activeIndex, setIndex, updateProfile } = useContext(ActiveProfileContext);
    const { setScreen } = useContext(ActiveScreenContext);
    const {
        constants: { type },
        locals: { common },
        profiles,
    } = useContext(AppDataContext);

    const profileCount = profiles.length;
    const [visibleProfileCount, setVisibleProfileCount] = useState(wasMoreClicked ? profileCount : 3);

    const profileListRef = useRef<HTMLDivElement>(null);
    const [profileListStyle, setProfileListStyle] = useState<React.CSSProperties>(() => ({
        maxHeight: calculatedHeight ? `${calculatedHeight}px` : 'auto',
    }));

    const onActivateProfile = useCallback(
        (index: number) => {
            setIndex(index);
            updateProfile(profiles[index]);
        },
        [setIndex, updateProfile, profiles],
    );

    const onNewProfileClick = useCallback(() => {
        setIndex(-1);
        updateProfile({}, true);
        setScreen('form');
    }, [setIndex, setScreen, updateProfile]);

    const onMoreProfileClick = useCallback(() => {
        setVisibleProfileCount(profileCount);
        metrika.params({
            moreProfiles: 'click',
        });
        wasMoreClicked = true;
    }, [profileCount]);

    const isBro = type === 'bro';
    const needNewProfileButton = isBro && profileCount <= 3;
    const needMoreButtonButton = isBro && !needNewProfileButton && profileCount !== visibleProfileCount;
    const isScrollingBlock = isBro && !needNewProfileButton && !needMoreButtonButton;

    // На маунт компонента получаем его текущую высоту блока,
    // чтобы сохраняить ее независимо от количества профлей
    useEffect(() => {
        // Вынос из текущего скоупа исполнения нужен для FF - не успевают отрендериться карточки пресетов и,
        // как следствие, высота списка вычисляется как 0px
        setTimeout(() => {
            if (profileListRef.current && !calculatedHeight) {
                calculatedHeight = profileListRef.current.clientHeight;
                setProfileListStyle({ maxHeight: calculatedHeight });
            }
        });
    }, [profiles]);

    return (
        <div
            ref={profileListRef}
            className={classNames(styles['profile-list'], isScrollingBlock && styles['profile-list_scrolling'])}
            style={profileListStyle}
        >
            {profiles.slice(0, visibleProfileCount).map((profile, index) => (
                <Profile
                    key={index}
                    index={index}
                    profile={profile}
                    isActive={activeIndex === index}
                    onActivate={onActivateProfile}
                />
            ))}
            {needNewProfileButton && (
                <ActionButton dataT="add-preset-button" onClick={onNewProfileClick}>
                    {common.new}
                </ActionButton>
            )}
            {needMoreButtonButton && (
                <ActionButton dataT="more-preset-button" onClick={onMoreProfileClick}>
                    {common.other}
                </ActionButton>
            )}
        </div>
    );
};
