import React, { useContext, useEffect, useRef } from 'react';
import { CSSTransition, TransitionGroup } from 'react-transition-group';
import { TRANSITION_ANIMATION_DURATION_MS } from '@src/constants';
import { removeClassNames } from '@client/util';
import { classNames } from '@common/util';
import { metrika } from '@common/metrika';
import { Card } from '@common/components/Card';
import { Form } from '@common/components/Form';
import { AppDataContext } from '@common/data/AppDataContext';
import { ActiveScreenContext } from '@common/data/ActiveScreenContext';

import styles from './visible-transition.css';

type VisibleTransitionProps = {
    className: string;
};

const transitionClasses = {
    enter: styles.enter,
    enterActive: styles['enter-active'],
    enterDone: styles['enter-done'],
    exit: styles.exit,
    exitActive: styles['exit-active'],
    exitDone: styles['exit-done'],
};

// Удаляем css-классы переходов, чтобы исключить абсолютное позиционирование и выставить актуальную высоту формы
const onEntered = (node: HTMLElement) => removeClassNames(node, [styles['enter-done']]);
const onExited = (node: HTMLElement) => removeClassNames(node, [styles['exit-done']]);

const Screen: React.FC<VisibleTransitionProps> = ({ className, children }) => (
    <div className={classNames(styles.transition, className)}>{children}</div>
);

export const ScreenManager: React.FC = () => {
    const screenContainerRef = useRef<HTMLDivElement>(null);

    const {
        meta: { targetOrigin },
        profiles,
    } = useContext(AppDataContext);
    const { screen } = useContext(ActiveScreenContext);
    const profilesCount = profiles.length;

    const isCardVisible = screen === 'card';
    const isFormVisible = screen === 'form';

    useEffect(() => {
        metrika.params({
            open: {
                url: targetOrigin,
                profilesCount: String(profilesCount),
            },
        });
    }, [targetOrigin, profilesCount]);

    // При смене уровня формы, во время перехода экраны абсолютно позиционируются.
    // Чтобы сохранить высоту контейнера, перед переходом принудительно выставляем ему фиксированную высоту
    // уровня, на который осуществляется переход
    useEffect(() => {
        const enteringNode = document.querySelector(`.${styles.enter}`);

        if (!screenContainerRef.current || !enteringNode) {
            return;
        }

        screenContainerRef.current.style.height = `${enteringNode.clientHeight}px`;

        // После перехода, убираем фиксированную высоту, чтобы была возможность динамически обновлять контент формы.
        // Добавляем дополнительное время, чтобы наверняка дождаться окончания перехода.
        const timerId = window.setTimeout(() => {
            if (screenContainerRef.current) {
                screenContainerRef.current.removeAttribute('style');
            }
        }, TRANSITION_ANIMATION_DURATION_MS + 200);

        return () => window.clearTimeout(timerId);
    }, [screen]);

    return (
        <div ref={screenContainerRef} className={styles['screen-container']}>
            <TransitionGroup component={null}>
                {isCardVisible && (
                    <CSSTransition
                        timeout={TRANSITION_ANIMATION_DURATION_MS}
                        classNames={transitionClasses}
                        onEntered={onEntered}
                        onExited={onExited}
                    >
                        <Screen className={styles.card}>
                            <Card />
                        </Screen>
                    </CSSTransition>
                )}
                {isFormVisible && (
                    <CSSTransition
                        timeout={TRANSITION_ANIMATION_DURATION_MS}
                        classNames={transitionClasses}
                        onEntered={onEntered}
                        onExited={onExited}
                    >
                        <Screen className={styles.form}>
                            <Form />
                        </Screen>
                    </CSSTransition>
                )}
            </TransitionGroup>
        </div>
    );
};
