import React, { useMemo, useCallback, useState, useRef, useEffect } from 'react';
import { useOutsideClick } from '@common/hooks/useOutsideClick';
import { classNames, calcTooltipDirection } from '@common/util';
import { Input } from '@common/components/Input';

import styles from './select.css';

export type SelectOption = {
    value: string;
    label: string;
    isNew?: boolean;
    checked?: boolean;
};

type SelectItemProps = {
    index: number;
    dataT: string;
    option: SelectOption;
    onSelect: (option: SelectOption, index: number) => void;
};

const inputHeight = 36;

const SelectItem: React.FC<SelectItemProps> = ({ dataT, index, option, onSelect }) => {
    const { value, isNew } = option;

    const onClick = useCallback(() => onSelect(option, index), [index, option, onSelect]);

    return (
        <div
            data-t={`${dataT}-item-${index}`}
            className={classNames(styles['tooltip-item'], isNew && styles['tooltip-item-new'])}
            onClick={onClick}
        >
            {value}
        </div>
    );
};

type SelectOptionsProps = {
    dataT: string;
    options: SelectOption[];
    onOptionSelect: (option: SelectOption, index: number) => void;
};

const Tooltip: React.FC<SelectOptionsProps> = ({ dataT, options, onOptionSelect }) => {
    const tooltipRef = useRef<HTMLDivElement>(null);
    const [tooltipStyles, setTooltipStyles] = useState<React.CSSProperties>({ maxHeight: 'auto' });

    useEffect(() => {
        if (tooltipRef.current) {
            const { length, direction } = calcTooltipDirection(tooltipRef.current);
            const bottom = direction === 'bottom' ? 0 : inputHeight;

            setTooltipStyles({
                visibility: 'visible',
                bottom: bottom ? `${bottom}px` : 'auto',
                maxHeight: length ? `${length - bottom}px` : 'auto',
            });
        }
    }, []);

    return (
        <div ref={tooltipRef} className={classNames(styles.tooltip)} style={tooltipStyles}>
            {options.map((option, index) => (
                <SelectItem dataT={dataT} key={option.value} index={index} option={option} onSelect={onOptionSelect} />
            ))}
        </div>
    );
};

type SelectProps = {
    label: string;
    dataT: string;
    options: SelectOption[];
    onOptionSelect: (option: SelectOption, index: number) => void;
    className?: string;
    newValue?: string;
    onNewValueChange?: (value: string) => void;
};

export const Select: React.FC<SelectProps> = props => {
    const { dataT, className, label, options, onOptionSelect } = props;

    const controlRef = useRef<HTMLDivElement>(null);
    const [isTooltipVisible, setTooltipVisible] = useState(false);

    const checkedOption = useMemo(() => options.filter(({ checked }) => checked)?.[0], [options]);

    const closeTooltip = useCallback(() => setTooltipVisible(false), []);

    const onTooltipItemClick = useCallback(
        (option: SelectOption, index: number) => {
            onOptionSelect(option, index);

            closeTooltip();
        },
        [onOptionSelect, closeTooltip],
    );

    const onInputClick = useCallback(() => {
        setTooltipVisible(visible => !visible);
    }, []);

    useOutsideClick([controlRef], closeTooltip);

    return (
        <div className={classNames(styles.select, className)}>
            <Input
                disabled
                dataT={dataT}
                theme="select"
                label={label}
                value={checkedOption?.label ?? ''}
                onClick={onInputClick}
                controlRef={controlRef}
            />
            {isTooltipVisible && <Tooltip dataT={dataT} options={options} onOptionSelect={onTooltipItemClick} />}
        </div>
    );
};
