import React, { useCallback, useContext, useState, useEffect, useRef } from 'react';

import { classNames } from '@common/util';
import { metrika } from '@common/metrikaSuggest';
import { AppDataContext } from '@common/data/AppDataContext';
import { sendMessageToParent } from '@client/util';
import { messages } from '@client/constants';
import { getBrowserManager } from '@common/browser-manager';
import { SuggestUser } from '../SuggestUser';
import { LoginButton } from '../ActionButtons/LoginButton';
import { PersonalButton } from '../PersonalButton/PersonalButton';

import styles from './index.css';

const WINDOW_WIDTH = 640;
const WINDOW_HEIGHT = 720;
const SUGGEST_INITIAL_HEIGHT = 189;
const SUGGEST_SHADOW_HEIGHT = 30;
const SUGGEST_MARGIN_TOP = 24;

export const SuggestCard: React.FC = () => {
    const { users = [], locals, constants: { suggestView = 'default' } = {} } = useContext(AppDataContext);
    const suggestRef = useRef<HTMLDivElement>(null);
    const [currentUserIndex, setCurrentUserIndex] = useState(0);
    const [isSuggestVisible, setSuggestVisibility] = useState(false);
    let popupUrl = `/suggest/popup${document.location.search}`;

    if (users.length) {
        popupUrl = `${popupUrl}&publicId=${users[currentUserIndex] && users[currentUserIndex].publicId}`;
    }

    const onLogInClick = useCallback(() => {
        metrika.params('click');
        window.popupWindow = window.open(
            popupUrl,
            'oauth',
            `left=${(screen.width - WINDOW_WIDTH) / 2}, top=${
                (screen.height - WINDOW_HEIGHT) / 2
            }, width=${WINDOW_WIDTH}, height=${WINDOW_HEIGHT}, scrollbars=no`,
        );
    }, [popupUrl]);
    useEffect(() => {
        if (suggestRef.current) {
            const height = suggestRef.current.scrollHeight || SUGGEST_INITIAL_HEIGHT;
            const frameHeight = ['button'].includes(suggestView)
                ? height
                : height + SUGGEST_SHADOW_HEIGHT + SUGGEST_MARGIN_TOP;
            sendMessageToParent({
                cause: 'suggest',
                type: messages.height,
                payload: {
                    height: frameHeight,
                },
            });
        }
    }, [isSuggestVisible, suggestView]);

    useEffect(() => metrika.params('open'), []);

    const renderDefaultSuggest = () => (
        <div className={classNames(styles.card)} ref={suggestRef} data-t="suggest">
            <div className={styles.header} data-t="suggest:header">
                <div className={styles.ya} />
                <div className={styles.title}>{locals.suggest['header.title']}</div>
                <button
                    className={styles.cross}
                    onClick={() => getBrowserManager().onClose()}
                    data-t="suggest:header:closer"
                />
            </div>
            <div className={classNames(styles.suggest)} data-t="suggest:main">
                {users.length > 1 ? (
                    <button
                        className={classNames(styles.user, styles.user_more)}
                        onClick={() => setSuggestVisibility(!isSuggestVisible)}
                        data-t="suggest:user:many"
                    >
                        <SuggestUser user={users[currentUserIndex]} />
                        <div
                            className={classNames(styles.arrow, isSuggestVisible && styles.arrow_opened)}
                            data-t={`suggest:user:arrow:${isSuggestVisible ? 'open' : 'close'}`}
                        />
                    </button>
                ) : (
                    <div className={classNames(styles.user)} data-t="suggest:user:one">
                        <SuggestUser user={users[currentUserIndex]} />
                    </div>
                )}
                <div
                    className={classNames(styles.users, isSuggestVisible && styles.users_opened)}
                    data-t="suggest:users"
                >
                    {users.map((user, index) => (
                        <button
                            className={styles.user}
                            key={user.publicId}
                            onClick={() => {
                                setCurrentUserIndex(index);
                                setSuggestVisibility(false);
                            }}
                            data-t={`suggest:users:user:${index}`}
                        >
                            <SuggestUser user={user} />
                            {index === currentUserIndex && (
                                <div className={classNames(styles.check)} data-t="suggest:users:user:check" />
                            )}
                        </button>
                    ))}
                </div>
            </div>
            <div className={styles.footer} data-t="suggest:footer">
                <LoginButton className={styles.button} onClick={onLogInClick} dataT="suggest:button">
                    {locals.common.continue}
                </LoginButton>
            </div>
        </div>);

    const renderButtonSuggest = () => (
        <div className={classNames(styles.card, styles.card_button)} ref={suggestRef} data-t="suggest">
            <PersonalButton onClick={onLogInClick} dataT="suggest:button" user={users[currentUserIndex]}>
                {locals.common.continue}
            </PersonalButton>
        </div>);

    const getSuggestByView = (suggestView = 'default') => {
        switch (suggestView) {
            case 'button':
                return renderButtonSuggest();
            default:
                return renderDefaultSuggest();
        }
    };

    return (
        getSuggestByView(suggestView)
    );
};
