import { debounce, fetch } from '@client/util';
import { useState, useEffect, useMemo, useRef } from 'react';

type SuggestResult = {
    lat: number;
    lon: number;
    name: string;
    type: string;
    kind: string;
    geoid: number;
    local: boolean;
    desc?: string | null;
};

type SuggestResponse = {
    status: 'ok' | 'error';
    part?: string;
    results?: SuggestResult[];
};

export const useAddressSuggest = (value: string) => {
    const [items, setItems] = useState<string[]>([]);
    const [isLoading, setLoading] = useState(false);

    // Сохраняем значение пользовательского ввода в ref, чтобы избежать прямой зависимости useMemo.
    // При получении данных от сервера, необходимо проверить, что это значение совпадает со значением запроса.
    // В противном случае ответ от сервера будет не актуален.
    const storedValue = useRef('');
    storedValue.current = value;

    const fetchAddresses = useMemo(
        () =>
            debounce((input: string) => {
                setLoading(true);

                fetch<SuggestResponse>(
                    '/api/suggest-geo',
                    { part: input },
                    {
                        onSuccess: ({ part, results }) => {
                            if (!part || !results?.length || part !== storedValue.current) {
                                return;
                            }

                            const suggestData = results.reduce<string[]>((addresses, item) => {
                                const address = !item.desc
                                    ? item.name
                                    : `${item.desc.split(', ').reverse().join(', ')}, ${item.name}`;

                                if (address.toLowerCase().trim() !== input.toLowerCase().trim()) {
                                    addresses.push(address);
                                }

                                return addresses;
                            }, []);

                            setItems(suggestData);
                        },
                        always: () => {
                            setLoading(false);
                        },
                    },
                );
            }, 300),
        [],
    );

    useEffect(() => {
        if (value) {
            fetchAddresses(value);
        } else {
            setItems([]);
        }
    }, [fetchAddresses, value]);

    return { isLoading, items };
};
