export const canUseDOM = () => {
    return typeof window !== 'undefined' && typeof window.document !== 'undefined';
};

export const classNames = (...classNames: Array<string | false | null | undefined>) => {
    return classNames.filter(Boolean).join(' ');
};

export const includesInArray = <T, V extends T>(array: T[], value: V): boolean => Boolean(~array.indexOf(value));

export const assign = (target: object, ...sources: (object | undefined | null)[]) => {
    sources.forEach(source => {
        if (source === undefined || source === null) {
            return;
        }

        const keys = Object.keys(source);

        keys.forEach(key => {
            const desc = Object.getOwnPropertyDescriptor(source, key);
            if (desc !== undefined && desc.enumerable) {
                target[key] = source[key];
            }
        });
    });

    return target;
};

const rootNode = canUseDOM() ? document.querySelector<HTMLDivElement>('#root') : null;

type Direction = 'top' | 'bottom';
type TooltipDirection = {
    direction: Direction;
    length?: number;
};

export const calcTooltipDirection = (node: HTMLElement): TooltipDirection => {
    if (!rootNode) {
        return { direction: 'bottom' };
    }

    const rootBounding = rootNode.getBoundingClientRect();
    const nodeBounding = node.getBoundingClientRect();

    const toTopLength = nodeBounding.top - rootBounding.top - 10;
    const toBottomLength = rootBounding.bottom - nodeBounding.top - 10;

    if (toBottomLength >= nodeBounding.height) {
        return { direction: 'bottom', length: toBottomLength };
    }

    // Вычитаем небольшое расстояние от нижнего края,
    // чтобы опции в селекте не прибивались к низу окна
    return toTopLength >= toBottomLength
        ? { direction: 'top', length: toTopLength }
        : { direction: 'bottom', length: toBottomLength };
};
