import { URL } from 'url';
import config from '../config';

import type { Request } from 'express';

export const getExperiments = async (req: Request) => {
    const experiments: { flags: string[]; boxes: string[] } = {
        flags: [],
        boxes: [],
    };
    const regionId = await req.services.geo.getRegionId();
    const yandexuid = req.query._ym_uid || req.query.ym_uid || '';
    const requestHeaders: Record<string, string | string[] | undefined> = {
        Host: req.headers.host,
        Cookie: `yandexuid=${yandexuid}`,
        'X-Forwarded-For-Y': req.headers['x-real-ip'],
        'X-Region-City-Id': regionId ? String(regionId) : undefined,
        'User-Agent': (req.headers['user-agent'] || '').replace(/\n/g, ''),
    };
    const testId = req.cookies['test-id'] || req.query['test-id'];
    const flagIds = req.cookies['flag-ids'] || req.query['flag-ids'];
    let uaasUrl: string | URL = 'http://uaas.search.yandex.net/yandexuid';

    if (flagIds && config.environment === 'dev') {
        experiments.flags = flagIds.split(';');
    }

    if (!/^\d+$/.test(req.cookies.yandexuid)) {
        req.log.info('Failed to request exps. Invalid yandexuid: %s', JSON.stringify(requestHeaders));
        return experiments;
    }

    if (testId) {
        if (config.environment !== 'prod' || (await req.services.geo.checkInternalNetwork())) {
            try {
                uaasUrl = new URL(uaasUrl);
                uaasUrl.search = `test-id=${testId}`;
                requestHeaders['X-Yandex-UAAS'] = 'testing';
            } catch (err) {
                req.log.warn({
                    err,
                    type: 'experiments',
                    code: 'test-id',
                });
            }
        }
    }

    try {
        const { statusCode, headers } =
            (await req.makeRequest(uaasUrl, {
                timeout: 160,
                headers: requestHeaders,
                retry: {
                    limit: 3,
                    maxRetryAfter: 100,
                    statusCodes: [500],
                },
            })) || {};

        if (statusCode === 200 && headers?.['x-yandex-expflags']) {
            req.log.info({
                type: 'experiments',
                msg: `Experiment status header was: ${headers['x-yandex-expflags']}`,
            });

            const experimentBoxesHeader = headers['x-yandex-expboxes'] || '';
            const flagsHeader = headers['x-yandex-expflags'] || '';

            (Array.isArray(flagsHeader) ? flagsHeader : flagsHeader.split(',')).forEach((header: string) => {
                try {
                    const buffer = Buffer.from(header, 'base64');
                    const parsed = JSON.parse(buffer.toString());

                    if (Array.isArray(parsed) && parsed[0] && parsed[0].HANDLER === 'YANDEXID') {
                        const flags =
                            (parsed[0].CONTEXT && parsed[0].CONTEXT.YANDEXID && parsed[0].CONTEXT.YANDEXID.flags) || [];

                        experiments.flags = experiments.flags.concat(flags);
                    }
                } catch (err) {
                    req.log.warn({
                        err,
                        type: 'experiments',
                        code: 'parse',
                    });
                }
            });

            experiments.boxes = !Array.isArray(experimentBoxesHeader)
                ? experimentBoxesHeader.split(';')
                : experimentBoxesHeader;
        }
    } catch (err) {
        req.log.error({
            err,
            code: 'get',
            type: 'experiments',
            msg: `Failed to request exps. Headers: ${JSON.stringify(requestHeaders)}`,
        });
    }

    return experiments;
};
