import got from 'got';

import type { URL } from 'url';
import type { Request } from 'express';
import type { Headers, Response, RequestError, CancelableRequest, OptionsOfJSONResponseBody } from 'got';

export type Options = OptionsOfJSONResponseBody & {
    getErrorLogData?: (err: RequestError) => Partial<{ headers: Headers }>;
};

export type RequestSender = <Data extends object>(
    url: string | URL,
    options?: Options,
) => CancelableRequest<Response<Data>>;

export const getHooks = (
    req: Request,
    getErrorLogData?: Options['getErrorLogData'],
): OptionsOfJSONResponseBody['hooks'] => ({
    beforeRequest: [
        options => {
            req.log.info({
                url: options.url,
                type: 'request',
                method: options.method || 'GET',
            });
        },
    ],
    beforeError: [
        err => {
            req.log.error({
                url: err.response?.requestUrl,
                type: 'error-response',
                err: {
                    name: err.name,
                    code: err.code,
                    message: err.message,
                },
                ...getErrorLogData?.(err),
            });

            return err;
        },
    ],
});

const makeRequest = (req: Request): RequestSender => <Data extends ANY = string>(
    url: string | URL,
    { getErrorLogData, ...options }: Options = {},
): CancelableRequest<Response<Data>> =>
    got<Data>(url, {
        ...options,
        hooks: getHooks(req, getErrorLogData),
    });

export default makeRequest;
