import { get } from 'psl';
import { Field, Constants } from '../../types';
import config from '../config';

import type { Request } from 'express';

type PassportValidateResponse = {
    status: 'ok' | 'error';
    permissions: 'all' | 'no_phone' | 'none';
    errors: string[];
};

type Options = {
    location: string;
    originHost: string;
    requestedFields: string[];
    action: Constants['type'];
};

type Result = {
    targetOrigin: string;
    whitelist: Field[];
};

const filterWitelist = (whitelist: Field[], requestedFields: string[]) => {
    if (whitelist.length && requestedFields.length) {
        return whitelist.filter(item => requestedFields.includes(item));
    }

    return whitelist;
};

export const validatePartner = async (req: Request, options: Options): Promise<Result> => {
    const { location, originHost, requestedFields, action } = options;

    // При локальной разработке нет доступа до апи паспорта, поэтому мокаем ответ
    if (config.local) {
        return {
            targetOrigin: `https://${req.hostname}`,
            whitelist: filterWitelist(['firstName', 'lastName', 'email', 'phone', 'address'], requestedFields),
        };
    }

    // Если форму запросил Браузер, то валидировать партнера не нужно
    if (action === 'bro') {
        return {
            targetOrigin: '',
            whitelist: filterWitelist(['firstName', 'lastName', 'email', 'phone', 'address'], requestedFields),
        };
    }

    if (action === 'suggest') {
        return {
            targetOrigin: `https://${originHost}`,
            whitelist: [],
        };
    }

    const { user, passport } = req.services;

    const { baseUrl } = config.passportAPI;

    const partnerId = String(req.query.partner_id || '') || get(originHost);
    const version = String(req.query.version || '');
    const requestOptions = passport.jsonRequestOptions;
    const searchParams = {
        consumer: 'autofill-frontend',
        partner_id: partnerId,
        page_url: location,
    };
    const passportLogData = {
        type: 'passport',
        action: 'backend',
        code: 'validate_partner',
        ym_uid: user.current?.yu,
    };

    let targetOrigin = '';
    let whitelist: Field[] = [];

    try {
        const {
            body: { status, errors, permissions },
        } = await req
            .makeRequest<PassportValidateResponse>(`${baseUrl}/1/bundle/autofill/validate_partner/`, {
                ...requestOptions,
                searchParams,
            })
            .catch(() => ({ body: { status: '', errors: [], permissions: '' } }));

        switch (permissions) {
            case 'all':
                targetOrigin = `https://${originHost}`;
                whitelist = ['firstName', 'lastName', 'email', 'phone', 'address'];
                break;
            case 'no_phone':
                targetOrigin = `https://${originHost}`;
                whitelist = ['firstName', 'lastName', 'email', 'address'];
                break;
            default:
                targetOrigin = '';
                whitelist = [];
        }

        if (config.environment === 'dev' && req.query.fake === '1' && !targetOrigin) {
            targetOrigin = `https://${req.hostname}`;
            whitelist = ['firstName', 'lastName', 'email', 'phone', 'address'];
        }

        whitelist = filterWitelist(whitelist, requestedFields);

        req.log.info({
            ...passportLogData,
            version,
            status,
        });
        req.log.info({
            action,
            version,
            errors,
            location,
            permissions,
            result: errors?.length ? 'not_available' : (!whitelist.length && 'no_data') || 'ready',
        });
    } catch (err) {
        req.log.error({
            ...passportLogData,
            version,
            err,
        });
    }

    return { targetOrigin, whitelist };
};
