import got from 'got';
import * as protobuf from 'protobufjs';
import { get as getSchemas } from '@yandex-int/maps-proto-schemas';
import config from '../../config';
import { validate } from '../../../util';
import { arrayOf } from '../../../helpers';
import { parseLocationFromGeoData } from './util';
import { getHooks } from '../../lib/requestFactory';

import type { RequestHandler } from 'express';
import type { GeoReply } from './types';
import type { FieldError, Field, APIBody, BundleDSResponseItem, MarketDSResponse } from '../../../types';

type BundleField = Exclude<Field, 'address'>;
type AddressField = 'country' | 'street' | 'city' | 'building';

const Response = protobuf
    .loadSync(getSchemas(['atom', 'common2', 'direct', 'search']))
    .lookupType('yandex.maps.proto.common2.response.Response');
const ADDRESS_ERROR_MAP: Record<AddressField, FieldError> = {
    city: 'address.required.city',
    street: 'address.required.street',
    country: 'address.required.country',
    building: 'address.required.building',
};
const ADDRESS_REQUIRED_FIELDS: AddressField[] = ['country', 'street', 'city', 'building'];

const handler: RequestHandler<Record<string, string>, APIBody['save']['res'], APIBody['save']['req']> = async (
    req,
    res,
) => {
    if (!/^\d{12,}$/.test(req.body.id || '')) {
        return res.json({ status: 'error' });
    }

    let addressData: ReturnType<typeof parseLocationFromGeoData> | null = null;
    const id = req.body.id;
    const cloudRequestOptions = req.services.cloud.requestOptions;
    const data: Record<Exclude<Field, 'address'>, string | null> = {
        firstName: null,
        lastName: null,
        email: null,
        phone: null,
    };
    const errors: Record<Field, FieldError | null> = {
        firstName: null,
        lastName: null,
        email: null,
        phone: null,
        address: null,
    };
    const dsLogData = {
        type: 'datasync',
        action: 'backend',
    };

    if (!cloudRequestOptions) {
        return res.json({ status: 'error', errors });
    }

    if (req.body.address && !req.tvm?.frontend.tickets?.geo?.ticket) {
        errors.address = 'internal';
    } else if (req.body.address) {
        await got(`${config.geo.baseUrl}/yandsearch`, {
            searchParams: {
                ms: 'pb',
                results: 1,
                type: 'geo',
                text: req.body.address,
                origin: config.geo.origin,
                lang: req.services.misc.locale,
            },
            responseType: 'buffer',
            timeout: config.geo.timeout,
            hooks: getHooks(req),
            headers: {
                'X-Ya-Service-Ticket': req.tvm?.frontend.tickets?.geo.ticket,
                'Content-type': 'application/json',
            },
        })
            .then(({ body }) => {
                const { error, reply } = Response.decode(body).toJSON() as { error: Error; reply: GeoReply };

                const location = error ? null : parseLocationFromGeoData(reply);

                if (!location) {
                    errors.address = 'address.invalid';
                    return;
                }

                for (const field of ADDRESS_REQUIRED_FIELDS) {
                    if (!location[field]) {
                        errors.address = ADDRESS_ERROR_MAP[field];
                        return;
                    }
                }

                addressData = location;
            })
            .catch(err => {
                req.log.error({
                    err,
                    type: 'geo',
                    code: 'geo-search',
                });
                errors.address = 'internal';
            });
    }

    arrayOf<BundleField>({
        firstName: -1,
        lastName: 0,
        phone: 1,
        email: 2,
    }).forEach(field => {
        const value = req.body[field];

        if (value) {
            const error = validate[field](value);

            if (error) {
                errors[field] = error;
            } else {
                data[field] = value;
            }
        }
    });

    if (Object.values(errors).some(Boolean)) {
        return res.json({ status: 'error', errors });
    }

    await Promise.all([
        new Promise(async resolve => {
            if (!addressData) {
                return resolve(false);
            }
            const cloudJsonRequestOptions = req.services.cloud.jsonRequestOptions;

            if (cloudJsonRequestOptions) {
                const {
                    body: { items },
                } = await req
                    .makeRequest<MarketDSResponse>(
                        `${config.cloud.baseUrl}/v1/personality/profile/market/delivery_addresses`,
                        cloudJsonRequestOptions,
                    )
                    .catch(() => ({ body: { items: [] } }));

                if (items.some(item => item.addressLine === addressData?.addressLine)) {
                    return resolve();
                }
            }

            await req
                .makeRequest(`${config.cloud.baseUrl}/v1/personality/profile/market/delivery_addresses/${id}`, {
                    ...cloudRequestOptions,
                    method: 'PUT',
                    json: addressData,
                })
                .catch(() => {
                    /* EMPTY */
                })
                .finally(resolve);
        }),
        Object.values(data).some(Boolean) || data.firstName || data.lastName
            ? new Promise(async resolve => {
                  const {
                      body: { items: bundleItems },
                  } = await req
                      .makeRequest<{ items: BundleDSResponseItem[] }>(
                          `${config.cloud.baseUrl}/v1/personality/profile/autofill/bundles`,
                          {
                              ...cloudRequestOptions,
                              responseType: 'json',
                          },
                      )
                      .catch(() => ({ body: { items: [] } }));
                  const { phones = [], emails = [], firstnames = [], lastnames = [] } =
                      bundleItems.find(({ id }) => id === '0') || {};

                  await req
                      .makeRequest(`${config.cloud.baseUrl}/v1/personality/profile/autofill/bundles/0`, {
                          ...cloudRequestOptions,
                          method: 'PUT',
                          json: {
                              phones:
                                  !data.phone || phones.some(item => item.value === data.phone)
                                      ? phones
                                      : [...phones, { id, name: '', value: data.phone }],
                              emails:
                                  !data.email || emails.some(item => item.value === data.email)
                                      ? emails
                                      : [...emails, { id, name: '', value: data.email }],
                              firstnames:
                                  !data.firstName || firstnames.some(item => item.value === data.firstName)
                                      ? firstnames
                                      : [...firstnames, { id, name: '', value: data.firstName }],
                              lastnames:
                                  !data.lastName || lastnames.some(item => item.value === data.lastName)
                                      ? lastnames
                                      : [...lastnames, { id, name: '', value: data.lastName }],
                          },
                      })
                      .catch(() => {
                          /*EMPTY */
                      })
                      .finally(resolve);
              })
            : Promise.resolve(false),
    ])
        .then(
            results =>
                !results.every(result => result === false) &&
                req.log.info({
                    ...dsLogData,
                    status: 'ok',
                }),
        )
        .catch(() =>
            req.log.error({
                ...dsLogData,
                status: 'error',
            }),
        );

    return res.json(Object.values(errors).some(Boolean) ? { status: 'error', errors } : { status: 'ok' });
};

export default handler;
