import * as path from 'path';
import * as express from 'express';
import cookieParser from 'cookie-parser';
import expressYandexuid from '@yandex-int/express-yandexuid';
import tvm from './tvm';
import config from '../config';
import error from './error';
import render from './render';
import params from './params';
import logger from './logger';
import request from './request';
import headers from './headers';
import uatraits from './uatraits';
import services from './services';
import blackbox from './blackbox';
import userMatch from './userMatch';
import langDetect from './langDetect';
import suggestPopup from './suggestPopup';
import { acceptDocumentsAgreement } from './acceptDocumentsAgreement';
import { socialStart, socialFinish, socialCsp } from './social';
import geoSuggest from './geo/suggest';
import cloudSaveData from './cloud/saveData';
import { generateToken, sendToken, checkToken } from './csrf';
import { providerFetchList } from './provider/list';

import type { Express } from 'express';
import { providerMiddlewares } from './provider';

const middleware = (app: Express) => {
    if (config.environment === 'dev') {
        app.use('/favicon.ico', (req, res) => res.status(404).end());
        app.use(config.paths.assets, express.static(path.resolve(__dirname, '../../../public/assets')));
        // TODO: remove next line, when path changes on sso
        app.use('/assets/autofill.js', express.static(path.resolve(__dirname, '../../../public/assets/sdk.js')));
    }

    app.disable('trust proxy');
    app.use(express.json());
    app.use(express.urlencoded({ extended: false }));
    app.use(cookieParser());
    app.use(params, logger, request, uatraits, expressYandexuid(), tvm, blackbox, langDetect, services);
    app.post('/api/token', sendToken);
    app.post('/api/suggest-geo', geoSuggest);
    app.post('/api/*', userMatch, checkToken);
    app.post('/api/save', cloudSaveData);
    app.post('/api/accept-documents-agreement', acceptDocumentsAgreement);
    app.post('*', (req, res) => res.json({ status: 'error' }));
    app.use(headers);
    app.get('*', generateToken);
    app.get('/embedded', render());
    app.get('/popup', render({ type: 'popup' }));
    app.get('/bro', render({ type: 'bro' }));
    app.get('/suggest', render({ type: 'suggest' }));
    app.get('/suggest/popup', suggestPopup);
    app.get('/provider', providerMiddlewares);
    app.get('/provider/api/list', providerFetchList, (req, res) => {
        res.json(res.locals.providerDocs);
    });
    app.get('/social/start', socialCsp, socialStart);
    app.get('/social/finish', socialCsp, socialFinish);
    app.get('*', (req, res) => res.redirect(req.services.misc.passportUrl));
    app.use(error);
};

export default middleware;
