import Time from '../lib/time';
import logger from '../lib/logger';

import type { Response, RequestHandler } from 'express';

const levelFromStatus = (statusCode: number) => {
    const key = String(Math.floor(statusCode / 100));
    const levels: Record<string, 'info' | 'warn' | 'error'> = {
        '1': 'warn',
        '2': 'info',
        '3': 'warn',
        '4': 'error',
        '5': 'error',
    };

    return levels[key] || 'error';
};

const handler: RequestHandler = (req, res, next) => {
    const time = new Time();

    req.log = logger.child({ requestId: req.id, ssrId: req.ssrId });
    req.log.info({
        type: 'request',
        method: req.method,
        url: req.originalUrl,
        yandexuid: req.cookies && req.cookies.yandexuid,
    });

    function onResFinished(this: Response) {
        const level = levelFromStatus(res.statusCode);
        const { raw } = req.blackbox || {};

        req.log[level]({
            type: 'response',
            ms: time.ms(),
            method: req.method,
            url: req.originalUrl,
            statusCode: res.statusCode,
            yandexuid: req.cookies && req.cookies.yandexuid,
            uid: raw && 'default_uid' in raw ? raw.default_uid : null,
        });

        this.removeListener('finish', onResFinished);
    }

    res.on('finish', onResFinished);

    return next();
};

export default handler;
