import React from 'react';
import serialize from 'serialize-javascript';
import { Request, Response } from 'express';
import { renderToStaticNodeStream } from 'react-dom/server';
import reactCDN from '@yandex-int/react-static-version/dist/version';
import config from '../../config';
import { Paranja, styles as paranjaStyles } from '../render/ssr/Paranja';
import type { User } from '../../../types';

type Props = {
    csrfToken: string;
    user: User | null;
    nonce: string;
    data: Object;
    constants: ANY;
    locals: Object;
    meta: Object;
    errors: Object;
};

const Html: React.FC<Props> = ({ children, nonce, data, constants, meta, locals, errors, user, csrfToken }) => {
    const isDev = config.environment === 'dev';
    const isDrawer = constants.isDrawer;

    return (
        <html>
            <head>
                <meta charSet="utf-8" />
                <meta
                    name="viewport"
                    content="width=device-width, initial-scale=1, maximum-scale=1, minimum-scale=1, shrink-to-fit=no, viewport-fit=cover"
                />
                <meta httpEquiv="X-UA-Compatible" content="ie=edge" />
                <link rel="shortcut icon" href="/favicon.ico" />
                <link href={`${config.paths.assets}/style-provider-render.css`} rel="stylesheet" />
                <script src={isDev ? reactCDN.development.reactWithDom : reactCDN.production.reactWithDom} />
                <style dangerouslySetInnerHTML={{ __html: paranjaStyles }} />
            </head>
            <body>
                <Paranja
                    offset={isDrawer ? 'none' : 'small'}
                    align={isDrawer ? 'bottom' : 'center'}
                    isOverflowHidden={false}
                    borderRadius={32}
                    isDrawer={isDrawer}
                >
                    <div id="root">{children}</div>
                </Paranja>
            </body>
            <script
                nonce={nonce}
                dangerouslySetInnerHTML={{
                    __html: `
                        window.__CSRF__ = ${serialize(csrfToken)};
                        window.__USER__ = ${serialize(user)};
                        window.__PROVIDER__ = ${serialize(data)};
                        window.__CONSTANTS__ = ${serialize(constants)}
                        window.__LOCALS__ = ${serialize(locals)};
                        window.__META__ = ${serialize(meta)}
                        window.__INITIAL_ERRORS__ = ${serialize(errors)}
                    `,
                }}
            />
            <script defer src={`${config.paths.assets}/provider-render.js`} />
        </html>
    );
};

const getErrorCode = (req: Request, res: Response) => {
    if (!res.locals.providerUser) {
        return 'no_user';
    } else if (!res.locals.providerDocs) {
        return 'no_docs';
    } else if (!res.locals.isYandexOrigin) {
        return 'bad_origin';
    } else if (!req.query.processName) {
        return 'no_process_name';
    }
};

export const renderProvider = (req: Request, res: Response) => {
    const {
        services: {
            misc: { tld, lang, passportAuthUrl, passportAuthUpdateUrl, locs },
            user,
        },
        query,
    } = req;
    const origin = String(query.origin);
    const isDrawer = Boolean(query.drawer === '1');
    const controlSum = req.id;

    res.set(
        'Content-Security-Policy',
        res
            .get('Content-Security-Policy')
            .replace(" frame-ancestors 'none'; ", ` frame-ancestors ${origin} *.yandex.ru:443; `),
    );
    res.append('Set-Cookie', `controlSum=${controlSum};Secure;SameSite=None`);
    res.type('html');

    const errorCode = getErrorCode(req, res);

    return renderToStaticNodeStream(
        <Html
            csrfToken={res.locals.csrfToken}
            nonce={req.nonce}
            data={{
                processName: String(query.processName),
                docs: res.locals.providerDocs,
                user: res.locals.providerUser,
            }}
            constants={{
                type: 'provider',
                dev: ['dev', 'testing'].includes(config.environment),
                tld,
                lang,
                isDrawer,
                passportAuthUrl,
                passportAuthUpdateUrl,
                avatarTemplate: config.avatar.url,
                ssrId: req.ssrId,
                metrikaId: config.metrika?.id,
            }}
            user={user.current}
            locals={locs}
            meta={{ targetOrigin: origin, whitelist: [], controlSum }}
            errors={errorCode ? { provider: errorCode } : {}}
        />,
    ).pipe(res);
};
