import { URL } from 'url';
import Clck from '@yandex-int/yandex-clck-url';
import config from '../../config';
import { getExperiments } from '../../lib/experiments';
import { getTheme } from '../../lib/theme';
import { validatePartner } from '../../lib/validatePartner';
import { ssrRender } from './ssr';

import type { RequestHandler } from 'express';
import type { RenderProps, SuggestUser } from '../../../types';

const metricsKey = Buffer.from('jQm9NqfREKGKRNgHSGLV1A==', 'base64');

type Constants = RenderProps['constants'];
type Settings = {
    type: Constants['type'];
};

const DEFAULT_SETTINGS: Settings = {
    type: 'embedded',
};

const mapUsers = (users: ExpressBlackbox.PassportUser[], defaultUid?: string | number): SuggestUser[] =>
    [...users]
        .sort((user1, user2) => {
            if (defaultUid) {
                if (user1.uid?.value === defaultUid) {
                    return -1;
                }
                if (user2.uid?.value === defaultUid) {
                    return 1;
                }
            }
            return 0;
        })
        .map(
            ({
                public_id: publicId,
                'address-list': adresses = [],
                display_name: { name = '', avatar: { default: avatar = '' } = {}, social = {} } = {},
                attributes = {},
            }) => ({
                publicId,
                name,
                avatar,
                defaultEmail: adresses.filter(adressInfo => adressInfo.default)[0]?.address,
                socialProvider: social.provider,
                hasPlus: attributes['1015'] === '1',
                firstName: attributes['27'] || '',
            }),
        );

const createHandler = (settings = DEFAULT_SETTINGS): RequestHandler => async (req, res) => {
    const {
        services: {
            misc: { tld, locs, lang, passportAuthUrl, passportAuthUpdateUrl },
            user,
        },
        blackbox,
    } = req;
    const location = String(req.query.location || '');
    const requestedFields = String(req.query.fields || '')
        .split(',')
        .filter(Boolean);
    const version = String(req.query.version || '');
    const theme = getTheme(req);
    const suggestView = String(req.query.suggest_view || 'default');

    let originHost: string = '';
    let referrerHost: string = '';

    try {
        const referrer = req.get('referrer') || '';

        if (referrer) {
            referrerHost = new URL(referrer).hostname;
        }
    } catch (err) {
        req.log.warn({
            err,
            version,
            type: 'render',
            code: 'referrer-origin-host',
        });
    }
    try {
        if (location) {
            originHost = new URL(location).hostname;
        }
    } catch (err) {
        req.log.warn({
            err,
            version,
            type: 'render',
            code: 'query-location-origin-host',
        });
    }

    if (requestedFields.includes('name')) {
        requestedFields.push('lastName', 'firstName');
    }

    if (settings.type === 'embedded' && referrerHost !== originHost) {
        req.log.error({
            err: new Error('BAD_REFERRER'),
            version,
            type: 'render',
            code: 'referrer-check',
        });
        return res.status(403).send('');
    }

    if (!['suggest', 'provider'].includes(settings.type)) {
        try {
            await user.enrichFromDS();
        } catch (err) {
            req.log.error({
                err,
                version,
                type: 'render',
                code: 'user-enrich',
            });
        }
    }

    const experiments = await getExperiments(req);
    const boxes = experiments.boxes || [];
    const flags = experiments.flags;
    const clck = new Clck({ key: metricsKey });
    const encodedBoxes = boxes ? clck.encrypt(boxes.join(';')) : '';
    const isDisableExp = flags.includes('autofill-disable-iframe-exp');

    const { targetOrigin, whitelist } = await validatePartner(req, {
        location,
        originHost,
        requestedFields,
        action: settings.type,
    });

    if (isDisableExp) {
        user.error = 'no_data';
        user.current = null;
    }

    const controlSum = req.id;

    // Для Бро нет нужды отправлять информацию о пользователе - вся информация придет из js-api
    // Для остального веба смотрим на результат валидации партнера
    const isUserEnabled = settings.type !== 'bro' && targetOrigin && whitelist.length;

    const users =
        ['suggest', 'provider'].includes(settings.type) && blackbox && blackbox.raw && 'users' in blackbox.raw
            ? mapUsers(blackbox.raw.users, blackbox.uid)
            : [];

    const renderProps: RenderProps = {
        csrfToken: res.locals.csrfToken,
        nonce: req.nonce,
        user: isUserEnabled ? user.current : null,
        users,
        profiles: isUserEnabled ? user.profiles : [],
        assetsPath: config.paths.assets,
        initialErrors: ['suggest', 'provider'].includes(settings.type)
            ? { users: users.length || ['button'].includes(suggestView) ? null : 'no_data' }
            : { user: user.error },
        meta: { whitelist, controlSum, targetOrigin },
        constants: {
            tld,
            passportAuthUrl,
            passportAuthUpdateUrl,
            avatarTemplate: config.avatar.url,
            lang,
            ssrId: req.ssrId,
            metrikaId: config.metrika?.id,
            type: settings.type,
            isDrawer: req.query.drawer === '1',
            experiments,
            encodedBoxes,
            theme,
            suggestView,
        },
        locals: locs,
    };

    if (['suggest'].includes(settings.type) && ['button'].includes(suggestView)) {
        renderProps.constants.suggestButtonOptions = {
            buttonSize: req.query.button_size && String(req.query.button_size) || 'm',
            buttonView: req.query.button_view && String(req.query.button_view) || 'main',
            buttonTheme: req.query.button_theme && String(req.query.button_theme) || 'light',
            buttonBorderRadius: req.query.button_border_radius && String(req.query.button_border_radius) || '0',
        };
    }

    if (['dev', 'testing'].includes(config.environment)) {
        renderProps.constants.dev = true;
    }

    if (targetOrigin !== '') {
        res.removeHeader('X-Frame-Options');
        res.set(
            'Content-Security-Policy',
            res
                .get('Content-Security-Policy')
                .replace(" frame-ancestors 'none'; ", ` frame-ancestors ${targetOrigin}; `),
        );
    }

    res.append('Set-Cookie', `controlSum=${controlSum};Secure;SameSite=None`);
    res.type('html');

    return ssrRender(res, renderProps);
};

export default createHandler;
