import type { RequestHandler } from 'express';
import { checkOrigin } from './checkOrigin';
import { getUser } from './getUser';
import { renderSocial } from './render';

export const socialStart: RequestHandler = async (req, res) => {
    const {
        query: { appName, scope, consumer, origin },
        services: { social },
    } = req;

    if (
        typeof appName !== 'string' ||
        typeof scope !== 'string' ||
        typeof consumer !== 'string' ||
        typeof origin !== 'string'
    ) {
        return res.end('bad_query');
    }

    if (!checkOrigin(origin)) {
        return res.end('bad_origin');
    }

    if (!getUser(req)) {
        return renderSocial(req, res, { origin, result: { state: 'error', code: 'no_user' } });
    }

    const { body } = await social.createTaskForProfile({ appName, scope, consumer });
    const { status, errors, task_id: taskId } = body || {};

    // Все плохо
    if (status === 'error' && errors) {
        req.log.warn({
            type: 'social',
            code: 'error',
            response: errors,
        });

        // Нет токена от есиа
        if (errors.includes('provider_token.not_found')) {
            // Отправляем в есиа
            return social.redirectToEsia(res, { scope, appName, origin });
        }
        // Что-то другое

        return renderSocial(req, res, { origin, result: { state: 'error', code: res.end(errors.join('; ')) } });
    }

    // Все хорошо
    if (taskId) {
        return res.redirect(`/social/finish?origin=${encodeURIComponent(origin)}&task_id=${taskId}`);
    }

    req.log.warn({
        type: 'social',
        code: 'no_task_id',
        response: body,
    });

    return renderSocial(req, res, { origin, result: { state: 'error', code: 'no_task_id' } });
};
