import locs from '../../locs';
import config from '../config';
import BaseService from './base';
import { arrayOf } from '../../helpers';

import type { Request } from 'express';
import type { Constants } from '../../types';

const LOCALES: Record<string, string> = { com: 'US', ru: 'RU', ua: 'UA', 'com.tr': 'TR' };
const LANGS = arrayOf<Constants['lang']>({
    ru: 0,
    en: 1,
});

export default class MiscService extends BaseService {
    url: string;
    tld: string;
    locale: string;
    lang: Constants['lang'];
    passportUrl: string;
    passportAuthUrl: string;
    passportAuthUpdateUrl: string;
    locs: typeof locs[Constants['lang']];

    constructor(req: Request) {
        super(req);

        const lang = this.req.langdetect.id;
        const domain = /^(?:.*\.)?yandex\.(.*)$/.exec(this.req.hostname);
        const url = `https://${this.req.get('host')}${this.req.originalUrl}`;
        const retpath = encodeURIComponent(url + '?open=1');

        this.url = url;
        this.lang = LANGS.includes(lang) ? lang : 'ru';
        this.tld = (domain !== null && domain[1]) || 'ru';
        this.locale = `${this.lang}_${LOCALES[this.tld] || LOCALES.ru}`;
        this.locs = locs[this.lang];
        this.passportUrl = config.passport.baseUrl;
        this.passportAuthUrl = `${config.passport.baseUrl}/auth/?retpath=${retpath}&noreturn=1&origin=autofill`;
        this.passportAuthUpdateUrl = `${config.passport.baseUrl}/auth/update/?retpath=${retpath}`;
    }
}
