import BaseService from './base';
import config from '../config';

import type { Request } from 'express';
import type { BaseMakeRequestResult } from '../../types';

const { retry, timeout, host, baseUrl } = config.passportAPI;

type Options = {
    timeout: number;
    retry: typeof retry;
    headers: Record<string, string>;
};

export default class PassportService extends BaseService {
    requestOptions: Options | null = null;
    jsonRequestOptions: (Options & { responseType: 'json' }) | null = null;

    constructor(req: Request) {
        super(req);

        if (this.req.tvm?.frontend.tickets?.passport?.ticket) {
            this.requestOptions = {
                retry,
                timeout,
                headers: {
                    Host: host,
                    'ya-consumer-client-ip': this.req.get('x-real-ip') || '',
                    'X-Ya-Service-Ticket': this.req.tvm.frontend.tickets.passport.ticket,
                    'Content-type': 'application/x-www-form-urlencoded',
                },
            };
        }

        if (this.req.blackbox?.userTicket && this.requestOptions) {
            this.requestOptions.headers['X-Ya-User-Ticket'] = this.req.blackbox?.userTicket;
        } else {
            req.log.error({
                requestOptions: this.requestOptions,
                status: 'error',
            });
        }
        if (this.requestOptions) {
            this.jsonRequestOptions = {
                ...this.requestOptions,
                responseType: 'json',
            };
        }
    }

    async acceptDocumentsAgreement() {
        return this.req.makeRequest<BaseMakeRequestResult>(`${baseUrl}/2/account/options/`, {
            ...this.jsonRequestOptions,
            method: 'POST',
            searchParams: { consumer: 'autofill-frontend' },
            form: {
                is_documents_agreement_accepted: 'yes',
            },
        });
    }
}
