import BaseService from './base';
import config from '../config';

import type { Request, Response } from 'express';

const { retry, timeout, host } = config.socialAPI;

type Options = {
    timeout: number;
    retry: typeof retry;
    headers: Record<string, string>;
};

export default class SocialService extends BaseService {
    requestOptions: Options | null = null;
    jsonRequestOptions: (Options & { responseType: 'json' }) | null = null;

    constructor(req: Request) {
        super(req);

        if (this.req?.blackbox?.userTicket && this.req.tvm?.frontend.tickets?.socialism?.ticket) {
            this.requestOptions = {
                retry,
                timeout,
                headers: {
                    Host: host,
                    'ya-consumer-client-ip': this.req.get('x-real-ip') || '',
                    'X-Ya-User-Ticket': this.req.blackbox.userTicket,
                    'X-Ya-Service-Ticket': this.req.tvm.frontend.tickets.socialism.ticket,
                    'Content-type': 'application/json',
                },
            };
            this.jsonRequestOptions = {
                ...this.requestOptions,
                responseType: 'json',
            };
        }
    }

    createTaskForProfile = async ({
        appName,
        scope,
        consumer,
    }: {
        appName: string;
        scope: string;
        consumer: string;
    }) => {
        const requestOptions = this.jsonRequestOptions;

        const { baseUrl } = config.socialAPI;

        return this.req.makeRequest<ANY>(`${baseUrl}/brokerapi/create_task_for_profile`, {
            ...requestOptions,
            method: 'POST',
            searchParams: { consumer: 'autofill-frontend', application_name: appName, scope, task_consumer: consumer },
            json: { application_name: appName, scope, task_consumer: consumer },
        });
    };

    getDataByTaskId = (taskId: string) => {
        const requestOptions = this.jsonRequestOptions;

        const { baseUrl } = config.socialAPI;

        return this.req.makeRequest<ANY>(`${baseUrl}/proxy2/task/${taskId}`, {
            ...requestOptions,
            method: 'GET',
            searchParams: { consumer: 'autofill-frontend' },
        });
    };

    redirectToEsia = (
        res: Response,
        { scope, appName, origin }: { scope: string; appName: string; origin: string },
    ) => {
        const { frontendUrl } = config.socialAPI;
        const retpath = `https://${this.req.hostname}/social/finish?origin=${encodeURIComponent(origin)}`;

        return res.redirect(
            `${frontendUrl}/broker2/start?provider=esia&application=${appName}&consumer=autofill-frontend&bind=1&scope=${scope}&place=query&retpath=${retpath}`,
        );
    };
}
