import * as punycode from 'punycode';
import escape from 'lodash/escape';
import BaseService from './base';
import config from '../config';
import { uniq } from '../../util';

import type { Request } from 'express';
import type { User, BundleDSResponseItem, MarketDSResponse, UserProfile } from '../../types';

const processEmail = (email: string) => {
    const parts = email.split('@');
    return escape([parts[0], punycode.toUnicode(parts[1])].join('@'));
};
const processAddress = ({ city, street, building }: MarketDSResponse['items'][number]) => {
    return [city, [street, building.split(' ').join(' ')].join(', ')].join(', ');
};
const getBlackboxEmailList = (bb: ExpressBlackbox.Result): ExpressBlackbox.UserEmail[] => {
    const { uid = '', raw } = bb;

    if ('address-list' in raw && raw['address-list']) {
        return raw['address-list'];
    }

    if ('users' in raw && raw.users) {
        return raw.users.find(({ uid: userUid }) => userUid && userUid.value === uid)?.['address-list'] ?? [];
    }

    return [];
};

export default class UserService extends BaseService {
    error: string | null = null;
    current: User | null = null;

    constructor(req: Request) {
        super(req);

        const bb = this.req.blackbox;

        if (!bb || !bb.error || typeof bb.uid !== 'string') {
            this.error = 'no_data';
        } else if (bb.error !== 'OK') {
            this.error = bb.error;
        } else if (bb.status !== 'VALID') {
            this.error = bb.status;
        }

        if (!bb || this.error) {
            return;
        }

        const { uid = '', avatar, phones, firstName, lastName } = bb;
        const avatarId = avatar?.default || config.avatar.defaultId;
        const avatarUrl = config.avatar.url.replace('%avatar_id%', avatarId);

        this.current = {
            uid,
            firstName,
            lastName,
            addresses: [],
            yu: this.req.cookies.yandexuid,
            phones: uniq(phones?.map(phone => phone.formatted) || []),
            emails: uniq(
                getBlackboxEmailList(bb).reduce<string[]>((acc, cur) => {
                    if (!cur.native || cur.default) {
                        const email = processEmail(cur.address);
                        acc.push(email);
                    }
                    return acc;
                }, []) || [],
            ),
            avatar: {
                id: avatarId,
                middle: avatarUrl.replace('%size%', 'middle'),
                retinaMiddle: avatarUrl.replace('%size%', 'retina-middle'),
            },
        };
    }

    async enrichFromDS() {
        if (!this.current) {
            return;
        }

        const { baseUrl } = config.cloud;
        const requestOptions = this.req.services.cloud.jsonRequestOptions;

        if (!requestOptions) {
            throw new Error('Cloud tvm ticket or user ticket not found');
        }

        await Promise.all([
            this.req
                .makeRequest<{ items: BundleDSResponseItem[] }>(
                    `${baseUrl}/v1/personality/profile/autofill/bundles`,
                    requestOptions,
                )
                .then(({ body: { items } }) => {
                    if (!this.current) {
                        return;
                    }

                    const bundleItem = items.find(({ id }) => id === '0');

                    if (bundleItem) {
                        this.current.phones = uniq(
                            this.current.phones.concat(...bundleItem.phones.map(({ value }) => value)),
                        );
                        this.current.emails = uniq(
                            this.current.emails.concat(...bundleItem.emails.map(({ value }) => value)),
                        );
                    }
                })
                .catch(() => {
                    /* EMPTY */
                }),
            this.req
                .makeRequest<MarketDSResponse>(
                    `${baseUrl}/v1/personality/profile/market/delivery_addresses`,
                    requestOptions,
                )
                .then(({ body: { items } }) => {
                    if (!this.current) {
                        return;
                    }

                    this.current.addresses = uniq(
                        this.current.addresses.concat(...items.map(item => processAddress(item))),
                    );
                })
                .catch(() => {
                    /* EMPTY */
                }),
        ]);
    }

    get profiles(): UserProfile[] {
        if (!this.current) {
            return [];
        }

        const { firstName, lastName, phones = [], addresses = [], emails = [] } = this.current;

        // TODO: https://st.yandex-team.ru/PASSP-31562 получать и сохранять профили из датасинка
        return [
            {
                firstName,
                lastName,
                phone: phones[0],
                email: emails[0],
                address: addresses[0],
            },
        ];
    }
}
