import type { UserProfile, Field, FieldError } from './types';

const MAX_LEN = 100;
const PHONE_REGEXP = /^\+7 \d{3} \d{3}‒\d{2}‒\d{2}$/;
const EMAIL_REGEXP = /(?:[a-z0-9!#$%&'*+/=?^_`{|}~-]+(?:\.[a-z0-9!#$%&'*+/=?^_`{|}~-]+)*|"(?:[\x01-\x08\x0b\x0c\x0e-\x1f\x21\x23-\x5b\x5d-\x7f]|\\[\x01-\x09\x0b\x0c\x0e-\x7f])*")@(?:(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?|\[(?:(?:(2(5[0-5]|[0-4][0-9])|1[0-9][0-9]|[1-9]?[0-9]))\.){3}(?:(2(5[0-5]|[0-4][0-9])|1[0-9][0-9]|[1-9]?[0-9])|[a-z0-9-]*[a-z0-9]:(?:[\x01-\x08\x0b\x0c\x0e-\x1f\x21-\x5a\x53-\x7f]|\\[\x01-\x09\x0b\x0c\x0e-\x7f])+)])/;

const getError = (value: string, validator?: (value: string) => FieldError | null) => {
    // if (!value) {
    //     return 'required';
    // }
    if (value.length > MAX_LEN) {
        return 'toolong';
    }

    return validator?.(value) || null;
};
export const validate: Record<Field, (value: string) => FieldError | null> = {
    firstName: (value: string) => getError(value),
    lastName: (value: string) => getError(value),
    phone: (value: string) => getError(value, () => (!PHONE_REGEXP.test(value) ? 'phone.invalid' : null)),
    email: (value: string) => getError(value, () => (!EMAIL_REGEXP.test(value) ? 'email.invalid' : null)),
    address: (value: string) => getError(value),
};
export const formatPhone = (value?: string) =>
    (value || '').replace(/\D/g, '').replace(/^[78]?(\d{3})(\d{3})(\d{2})(\d{2})$/, '+7 $1 $2‒$3‒$4');
export const createId = () => String(Math.round(Math.random() * 1e16));
export const uniq = <T>(arr: T[]) =>
    arr.reduce<T[]>((acc, cur) => {
        if (!~acc.indexOf(cur)) {
            acc.push(cur);
        }

        return acc;
    }, []);

const profileKeys: (keyof UserProfile)[] = ['firstName', 'lastName', 'email', 'phone', 'address'];
const isProfilesEqual = (a: UserProfile, b: UserProfile) => {
    for (let fieldName of profileKeys) {
        if (a[fieldName] !== b[fieldName]) {
            return false;
        }
    }

    return true;
};

const sanitizeProfile = (profile: UserProfile, whitelist: Field[]): UserProfile | null => {
    return whitelist.reduce<UserProfile | null>((prepareProfile, field) => {
        if (profile[field]) {
            if (!prepareProfile) {
                prepareProfile = {};
            }

            prepareProfile[field] = profile[field];
        }

        return prepareProfile;
    }, null);
};

export const filterProfilesByWhitelist = (profiles: UserProfile[], whitelist: Field[]): UserProfile[] => {
    if (!whitelist.length) {
        return [];
    }

    return profiles.reduce<UserProfile[]>((sanitizedProfiles, profile) => {
        const sanitizedProfile = sanitizeProfile(profile, whitelist);

        if (sanitizedProfile) {
            const isProfileUnique = sanitizedProfiles.every(existProfile => {
                return !isProfilesEqual(existProfile, sanitizedProfile);
            });

            if (isProfileUnique) {
                sanitizedProfiles.push(sanitizedProfile);
            }
        }

        return sanitizedProfiles;
    }, []);
};
