const path = require('path');
const webpack = require('webpack');
const MiniCssExtractPlugin = require('mini-css-extract-plugin');
const TsconfigPathsPlugin = require('tsconfig-paths-webpack-plugin');

const VERSION = `'${require('./package.json').version}'`;

class MiniCssExtractPluginCleanup {
    shouldPickStatChild(child) {
        return child.name.indexOf('mini-css-extract-plugin') !== 0;
    }

    apply(compiler) {
        compiler.hooks.done.tap('CleanUpStatsPlugin', stats => {
            const children = stats.compilation.children;
            if (Array.isArray(children)) {
                // eslint-disable-next-line no-param-reassign
                stats.compilation.children = children.filter(child => this.shouldPickStatChild(child));
            }
        });
    }
}

function createEntry(file) {
    return [
        'regenerator-runtime',
        file,
    ];
}

const cssRegex = /\.css$/;
const nodeModulesRegex = /node_modules/;
const publicPath = path.resolve(__dirname, 'public');

const client = ({ production } = {}) => {
    return {
        entry: {
            ['bro-render']: createEntry(path.resolve(__dirname, 'src/client/bro-render.tsx')),
            ['suggest-render']: createEntry(path.resolve(__dirname, 'src/client/suggest-render.tsx')),
            ['iframe-render']: createEntry(path.resolve(__dirname, 'src/client/iframe-render.tsx')),
            ['provider-render']: createEntry(path.resolve(__dirname, 'src/client/provider-render.tsx')),
            ['social-render']: createEntry(path.resolve(__dirname, 'src/client/social-render.tsx')),
        },
        externals: {
            react: 'React',
            'react-dom': 'ReactDOM',
        },
        target: ['web', 'es5'],
        output: {
            filename: '[name].js',
            path: path.resolve(publicPath, 'assets'),
        },
        resolve: {
            extensions: ['.ts', '.tsx', '.js'],
            plugins: [new TsconfigPathsPlugin()],
        },
        plugins: [
            new MiniCssExtractPlugin({
                ignoreOrder: true,
                filename: 'style-[name].css',
            }),
            new MiniCssExtractPluginCleanup(),
        ],
        module: {
            rules: [
                {
                    test: /\.ts(x?)$/,
                    exclude: nodeModulesRegex,
                    use: 'babel-loader',
                },
                {
                    test: cssRegex,
                    exclude: nodeModulesRegex,
                    use: [
                        MiniCssExtractPlugin.loader,
                        {
                            loader: 'css-loader',
                            options: {
                                importLoaders: production ? 1 : 0,
                                modules: {
                                    localIdentName: `${production ? '' : '[local]--'}[hash:base64]`,
                                },
                            },
                        },
                        ...(production
                            ? [
                                  {
                                      loader: 'postcss-loader',
                                  },
                              ]
                            : []),
                    ],
                    // Don't consider CSS imports dead code even if the
                    // containing package claims to have no side effects.
                    // Remove this when webpack adds a warning or an error for this.
                    // See https://github.com/webpack/webpack/issues/6571
                    sideEffects: true,
                },
                {
                    test: /\.(jpe?g|png|ttf|gif|eot|svg|woff(2)?)(\?[a-z0-9=&.]+)?$/,
                    issuer: cssRegex,
                    use: [
                        {
                            loader: 'base64-inline-loader',
                        },
                    ],
                },
            ],
        },
    };
};

const scriptCommon = ({ polyfills: POLYFILLS, entry, filename, production }) => ({
    entry: path.resolve(__dirname, `src/client/${entry}.ts`),
    output: {
        filename: `${filename}${POLYFILLS ? '-with-polyfills' : ''}-latest.js`,
        path: path.resolve(publicPath, 'scripts/v1'),
    },
    target: ['web', 'es5'],
    resolve: {
        extensions: ['.js', '.ts'],
        plugins: [new TsconfigPathsPlugin()],
    },
    plugins: [
        new webpack.DefinePlugin({ POLYFILLS, VERSION }),
        new MiniCssExtractPlugin({
            ignoreOrder: true,
            filename: `${filename}-latest.css`,
        }),
        new MiniCssExtractPluginCleanup(),
    ],
    module: {
        rules: [
            {
                test: /\.ts$/,
                use: 'babel-loader',
            },
            {
                test: cssRegex,
                exclude: nodeModulesRegex,
                use: [
                    MiniCssExtractPlugin.loader,
                    { loader: 'css-loader' },
                    ...(production ? [{ loader: 'postcss-loader' }] : []),
                ],
                // Don't consider CSS imports dead code even if the
                // containing package claims to have no side effects.
                // Remove this when webpack adds a warning or an error for this.
                // See https://github.com/webpack/webpack/issues/6571
                sideEffects: true,
            },
            {
                test: /\.(jpe?g|png|ttf|gif|eot|svg|woff(2)?)(\?[a-z0-9=&.]+)?$/,
                issuer: cssRegex,
                use: [{ loader: 'base64-inline-loader' }],
            },
        ],
    },
});

const scriptAutofill = (polyfills = false) => scriptCommon({ polyfills, entry: 'autofill', filename: 'sdk' });
const scriptSuggest = (polyfills = false) => scriptCommon({ polyfills, entry: 'suggest', filename: 'sdk-suggest' });
const scriptSuggestToken = (polyfills = false) =>
    scriptCommon({ polyfills, entry: 'token', filename: 'sdk-suggest-token' });
const scriptProvider = (polyfills = false) => scriptCommon({ polyfills, entry: 'provider', filename: 'sdk-provider' });
const scriptSocial = (production = false, polyfills = false) =>
    scriptCommon({ production, polyfills, entry: 'social', filename: 'sdk-social' });

module.exports = (params = {}, ...rest) => [
    ...(params['script-only'] ? [] : [client(params, ...rest)]),
    scriptAutofill(),
    scriptAutofill(true),
    scriptSuggest(),
    scriptSuggest(true),
    scriptSuggestToken(),
    scriptSuggestToken(true),
    scriptProvider(),
    scriptProvider(true),
    scriptSocial(params.production),
    scriptSocial(params.production, true),
];
